using System;

namespace Orciid.Core
{
	/// <summary>
	/// General exception in core
	/// </summary>
	/// <remarks>
	/// All exceptions thrown in the core should be instances of this class or derived classes.
	/// In the future, logging might be added to this class.
	/// </remarks>
	public class CoreException:
		System.ApplicationException
	{
		/// <summary>
		/// Constructor
		/// </summary>
		/// <remarks>
		/// The default constructor
		/// </remarks>
		/// <param name="text">An error message describing the circumstances causing
		/// the exception</param>
		public CoreException(string text):
			base(text)
		{
		}
	}

	/// <summary>
	/// Too many results exception
	/// </summary>
	/// <remarks>
	/// This exception is thrown when a search finds too many results.
	/// </remarks>
	public class TooManySearchResultsException:
		CoreException
	{
		/// <summary>
		/// Constructor
		/// </summary>
		/// <remarks>
		/// The default constructor
		/// </remarks>
		/// <param name="text">An error message describing the circumstances causing
		/// the exception</param>
		public TooManySearchResultsException(string text):
			base(text)
		{
		}
	}

	/// <summary>
	/// Validation error during import
	/// </summary>
	/// <remarks>
	/// This exception is thrown by the Import.Import method.
	/// </remarks>
	public class ImportValidationException:
		CoreException
	{
		/// <summary>
		/// Constructor
		/// </summary>
		/// <remarks>
		/// The default constructor
		/// </remarks>
		/// <param name="text">An error message describing the circumstances causing
		/// the exception</param>
		public ImportValidationException(string text):
			base(text)
		{
		}
	}
	
	/// <summary>
	/// Exception caused by insufficient or denied privileges
	/// </summary>
	/// <remarks>
	/// Whenever a restricted method is called, the current user's privileges are checked.
	/// If the privileges are insufficient, a PrivilegeException is raised.
	/// </remarks>
	public class PrivilegeException:
		CoreException
	{
		/// <summary>
		/// Required privileges
		/// </summary>
		/// <remarks>
		/// The privileges required to execute method in which the exception was raised
		/// </remarks>
		public Privilege RequiredPrivilege = Privilege.Unknown;
		
		/// <summary>
		/// Available privileges
		/// </summary>
		/// <remarks>
		/// The privileges available to the current user when the exception was raised
		/// </remarks>
		public Privilege AvailablePrivilege = Privilege.Unknown;

		/// <summary>
		/// Constructor
		/// </summary>
		/// <remarks>
		/// The default constructor
		/// </remarks>
		/// <param name="text">An error message describing the circumstances causing
		/// the exception</param>
		public PrivilegeException(string text):
			base(text)
		{
		}

		/// <summary>
		/// Constructor
		/// </summary>
		/// <remarks>
		/// Constructor
		/// </remarks>
		/// <param name="text">An error message describing the circumstances causing
		/// the exception</param>
		/// <param name="available">The privileges available to the current user
		/// when the exception was raised</param>
		/// <param name="required">The privileges that would have been required when
		/// the exception was raised</param>
		public PrivilegeException(string text, Privilege required, Privilege available):
			base(text)
		{
			RequiredPrivilege = required;
			AvailablePrivilege = available;
		}
	}
}
