using System;
using System.IO;
using NUnit.Framework;
using Orciid.Media.Converters;

namespace Orciid.Media
{
	[TestFixture]
	public class TestMediaConverter
	{
		static FileConverter converter = new FileConverter(DummyConvertFile);

		[SetUp]
		public void Setup()
		{
			MimeType.RegisterExtension("jpeg", "image/jpeg");
			MimeType.RegisterExtension("jpg", "image/jpeg");
			MimeType.RegisterExtension("png", "image/png");
			MimeType.RegisterExtension("dummy", "image/dummy");		
		}

		[TearDown]
		public void Cleanup()
		{
			MediaConverter.UnRegisterFileConverter(converter);
		}

		[Test]
		public void GetMimeType()
		{
			Assert.AreEqual("image/jpeg", MimeType.GetFromExtension(".jpg"));
			Assert.AreEqual("image/jpeg", MimeType.GetFromExtension(".jpeg"));
			Assert.AreEqual("image/jpeg", MimeType.GetFromExtension("image.jpeg"));
			Assert.AreEqual("image/jpeg", MimeType.GetFromExtension("jpeg"));
			Assert.AreEqual("image/png", MimeType.GetFromExtension(".png"));
			Assert.AreEqual("application/octet-stream", MimeType.GetFromExtension(""));
			Assert.AreEqual("application/octet-stream", MimeType.GetFromExtension("unknown.extension"));
		}

		[Test]
		public void ConverterRegistration()
		{			
			Assert.IsNull(MediaConverter.GetFileConverter("image/test-1", "image/test-2"));
			MediaConverter.RegisterFileConverter("image/test-1", "image/test-2", 
				Capabilities.None, true, converter);
			Assert.AreEqual(converter, MediaConverter.GetFileConverter("image/test-1", "image/test-2"));
			MediaConverter.UnRegisterFileConverter(converter);
			Assert.IsNull(MediaConverter.GetFileConverter("image/test-1", "image/test-2"));
		}

		[Test]
		public void ConvertFile()
		{
			MediaConverter.RegisterFileConverter("image/jpeg", "image/dummy", 
				Capabilities.None, true, converter);
			string tempfile1 = Path.GetTempFileName();
			File.Move(tempfile1, tempfile1 + ".jpg");
			try
			{
				Assert.IsTrue(MediaConverter.Convert(tempfile1 + ".jpg", tempfile1 + ".dummy"));
			}
			finally
			{
				File.Delete(tempfile1 + ".jpg");
				File.Delete(tempfile1 + ".dummy");
			}
		}

		private static bool DummyConvertFile(string sourcefile, string targetfile,
			Parameters parameters)
		{
			return true;
		}
	}
}
