using System;
using System.IO;
using System.Collections;
using System.Collections.Specialized;
using Org.Mentalis.Security.Certificates;

namespace Org.Mentalis.Security.Testing {
	public class CertificateBox : BlackBox {
		public CertificateBox(Stream fs) : base(fs) {}
		public override string Name {
			get {
				return "Certificate";
			}
		}
		public override int Start() {
			int tests = 0;
			Certificate c;
			/*
			 *  test Certificate constructors
			 */
			tests += TestConstructors();
			/*
			 *  load DER encoded server certificate
			 */
			try {
				c = CertificateStore.CreateFromCerFile(@"certs\server.der.cer").FindCertificateByUsage(new string[] {"1.3.6.1.5.5.7.3.1"});
				if (c == null) {
					AddError("C-S01");
				} else {
					tests += CheckServerCertBasicProps(c, true);
				}
			} catch {
				AddError("C-S03");
			}
			tests+=2;
			/*
			 *  load Base64 encoded server certificate
			 */
			try {
				c = CertificateStore.CreateFromCerFile(@"certs\server.base64.cer").FindCertificateByUsage(new string[] {"1.3.6.1.5.5.7.3.1"});
				if (c == null) {
					AddError("C-S04");
				} else {
					tests += CheckServerCertBasicProps(c, true);
				}
			} catch {
				AddError("C-S06");
			}
			tests+=2;
			/*
			 *  load PKCS#7 encoded server certificate
			 */
			try {
				c = CertificateStore.CreateFromCerFile(@"certs\server.p7b").FindCertificateByUsage(new string[] {"1.3.6.1.5.5.7.3.1"});
				if (c == null) {
					AddError("C-S07");
				} else {
					tests += CheckServerCertBasicProps(c, true);
				}
			} catch {
				AddError("C-S09");
			}
			tests+=2;
			/*
			 *  load PFX encoded server certificate
			 */
			try {
				c = CertificateStore.CreateFromPfxFile(@"certs\server.pfx", "test").FindCertificateByUsage(new string[] {"1.3.6.1.5.5.7.3.1"});
				if (c == null) {
					AddError("C-S10");
				} else {
					tests += CheckServerCertBasicProps(c, true);
					if (!c.HasPrivateKey())
						AddError("C-S11");
				}
			} catch {
				AddError("C-S12");
			}
			tests+=3;
			/*
			 *  load certificate [without private key]
			 */
			try {
				c = CertificateStore.CreateFromCerFile(@"certs\expired.cer").FindCertificateByUsage(new string[] {"1.3.6.1.5.5.7.3.1"});
				if (c == null) {
					AddError("C-S2");
				} else {
					if (c.HasPrivateKey())
						AddError("C-S5");
				}
			} catch {
				AddError("C-S8");
			}
			tests+=3;
			/*
			 *  test Certificate.ToX509()
			 */
			try {
				c = CertificateStore.CreateFromCerFile(@"certs\server.base64.cer").FindCertificateByUsage(new string[] {"1.3.6.1.5.5.7.3.1"});
				if (!c.Equals(c.ToX509()))
					AddError("C-S13");
			} catch {
				AddError("C-S14");
			}
			tests += 2;
			return tests;
		}
		private int TestConstructors() {
			Certificate c;
			try {
				c = new Certificate(null);
				AddError("C-TC1");
			} catch (ArgumentNullException) {
			} catch {
				AddError("C-TC2");
			}
			try {
				c = new Certificate(IntPtr.Zero);
				AddError("C-TC3");
			} catch (ArgumentException) {
			} catch {
				AddError("C-TC4");
			}
			try {
				c = new Certificate(IntPtr.Zero, true);
				AddError("C-TC5");
			} catch (ArgumentException) {
			} catch {
				AddError("C-TC6");
			}
			try {
				c = new Certificate(IntPtr.Zero, false);
				AddError("C-TC7");
			} catch (ArgumentException) {
			} catch {
				AddError("C-TC8");
			}
			try {
				Certificate c2 = null;
				try {
					c2 = CertificateStore.CreateFromCerFile(@"certs\server.der.cer").FindCertificateByUsage(new string[] {"1.3.6.1.5.5.7.3.1"});
				} catch {
					AddWarning("C-W-TC1");
				}
				if (c2 != null) {
					c = new Certificate(c2.Handle, true);
					if (!c.Equals(c2))
						AddError("C-TC9");
				}
			} catch {
				AddError("C-TC10");
			}
			try {
				Certificate c2 = null;
				try {
					c2 = CertificateStore.CreateFromCerFile(@"certs\server.der.cer").FindCertificateByUsage(new string[] {"1.3.6.1.5.5.7.3.1"});
				} catch {
					AddWarning("C-W-TC2");
				}
				if (c2 != null) {
					c = new Certificate(c2);
					if (!c.Equals(c2))
						AddError("C-TC11");
				}
			} catch {
				AddError("C-TC12");
			}
			return 12;
		}
		private int CheckServerCertBasicProps(Certificate c, bool clone) {
			byte[] SHA1Hash = new byte[] {0xE1, 0x4D, 0x2F, 0x99, 0x4E, 0x31, 0xA3, 0x2E, 0x5E, 0xD6, 0x83, 0x0A, 0x87, 0x59, 0x6F, 0x5E, 0x60, 0x5F, 0x33, 0x89};
			byte[] MD5Hash = new byte[] {0xAE, 0x8B, 0xA3, 0x7A, 0xF1, 0x1C, 0x33, 0x86, 0xFD, 0xCF, 0xC5, 0x23, 0xFF, 0x28, 0x2C, 0x6D};
			byte[] HashCode = new byte[] {0xE1, 0x4D, 0x2F, 0x99};
			byte[] Serial = new byte[] {0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD9, 0x72, 0xD2, 0x61};
			byte[] RawData = new byte[] {0x30, 0x82, 0x05, 0xF4, 0x30, 0x82, 0x03, 0xDC, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x0A, 0x61, 0xD2, 0x72, 0xD9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00,0x30, 0x2D, 0x31, 0x2B, 0x30, 0x29, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x22, 0x4D, 0x65, 0x6E, 0x74, 0x61, 0x6C, 0x69, 0x73, 0x2E, 0x6F, 0x72, 0x67, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75,0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x32, 0x31, 0x32, 0x30, 0x39, 0x31, 0x39, 0x34, 0x39, 0x35, 0x36, 0x5A, 0x17, 0x0D, 0x30, 0x33, 0x31, 0x32, 0x30, 0x39, 0x31, 0x39, 0x35, 0x39, 0x35, 0x36, 0x5A, 0x30,0x81, 0xA9, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x42, 0x45, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x07, 0x42, 0x72, 0x61, 0x62, 0x61, 0x6E, 0x74, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55,0x04, 0x07, 0x13, 0x06, 0x4C, 0x65, 0x75, 0x76, 0x65, 0x6E, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x0C, 0x54, 0x68, 0x65, 0x20, 0x4B, 0x50, 0x44, 0x2D, 0x54, 0x65, 0x61, 0x6D, 0x31, 0x1D, 0x30, 0x1B, 0x06, 0x03, 0x55,0x04, 0x0B, 0x13, 0x14, 0x53, 0x6F, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x44, 0x65, 0x76, 0x65, 0x6C, 0x6F, 0x70, 0x6D, 0x65, 0x6E, 0x74, 0x31, 0x1A, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x11, 0x4D, 0x65, 0x6E, 0x74, 0x61,0x6C, 0x69, 0x73, 0x2E, 0x6F, 0x72, 0x67, 0x20, 0x54, 0x65, 0x61, 0x6D, 0x31, 0x25, 0x30, 0x23, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x16, 0x77, 0x65, 0x62, 0x6D, 0x61, 0x73, 0x74, 0x65, 0x72, 0x40, 0x6D,0x65, 0x6E, 0x74, 0x61, 0x6C, 0x69, 0x73, 0x2E, 0x6F, 0x72, 0x67, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00,0xBC, 0x7B, 0x73, 0xD8, 0xF2, 0xA4, 0x8B, 0x17, 0x68, 0x4E, 0x49, 0xA1, 0x42, 0x52, 0x52, 0xFD, 0x1D,
			 0x83, 0x54, 0xE2, 0xB3, 0x20, 0x2B, 0x79, 0xBE, 0x57, 0x64, 0xEF, 0x2A, 0xDD, 0x6D, 0x41, 0x3A, 0xE8, 0xF6, 0xE3, 0x27, 0xFB, 0xFC, 0x4F,0xB3, 0xF2, 0x74, 0x35, 0x7F, 0xEE, 0x74, 0x78, 0xBB, 0x28, 0xAE, 0x58, 0x97, 0x5F, 0x73, 0xBE, 0x51, 0x0F, 0x49, 0xE4, 0x54, 0x56, 0x65, 0xB3, 0xB3, 0x44, 0x3F, 0xD5, 0x2C, 0x27, 0xA0, 0xDF, 0x6B, 0x6D, 0x8A, 0x3A, 0xAB, 0x5A, 0x42, 0xC3,0xDE, 0xC1, 0x4C, 0xB9, 0xA8, 0x59, 0x55, 0xC5, 0x7A, 0x99, 0x66, 0x1F, 0xB9, 0x9F, 0xE0, 0x83, 0x63, 0xE4, 0x60, 0xBE, 0xFF, 0xFF, 0x33, 0xC4, 0xB6, 0xF9, 0xF8, 0x7B, 0x12, 0xFD, 0x6F, 0x5D, 0x7F, 0x2A, 0x8A, 0xEF, 0x6E, 0xFA, 0x2D, 0xA5,0xFA, 0xF7, 0xEB, 0xD7, 0x2B, 0x8B, 0x87, 0x17, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x82, 0x02, 0x1B, 0x30, 0x82, 0x02, 0x17, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x04, 0x30, 0x30, 0x44, 0x06,0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x0F, 0x04, 0x37, 0x30, 0x35, 0x30, 0x0E, 0x06, 0x08, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x03, 0x02, 0x02, 0x02, 0x00, 0x80, 0x30, 0x0E, 0x06, 0x08, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,0x03, 0x04, 0x02, 0x02, 0x00, 0x80, 0x30, 0x07, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x07, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x03, 0x07, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0x37, 0x97,0x8A, 0x3D, 0x71, 0x3E, 0x24, 0xB6, 0xD8, 0x41, 0x61, 0x04, 0x0E, 0xFA, 0xA3, 0x45, 0xBF, 0x08, 0x90, 0x02, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30,0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xBF, 0x17, 0x6D, 0x60, 0xDA, 0x56, 0xCF, 0x36, 0xE4, 0x21, 0xF3, 0x51, 0x8A, 0x53, 0x64, 0x6C, 0xA1, 0x45, 0x61, 0x78, 0x30, 0x81, 0xA0, 0x06, 0x03, 0x55, 0x1D, 0x1F,0x04, 0x81, 0x98, 0x30, 0x81, 0x95, 0x30, 0x81, 0x92, 0xA0, 0x81, 0x8F, 0xA0, 0x81, 0x8C, 0x86, 0x45, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x6D, 0x61, 0x74, 0x68, 0x69, 0x6C, 0x64, 0x65, 0x2F, 0x43, 0x65, 0x72, 0x74, 0x45,
			 0x6E, 0x72,0x6F, 0x6C, 0x6C, 0x2F, 0x4D, 0x65, 0x6E, 0x74, 0x61, 0x6C, 0x69, 0x73, 0x2E, 0x6F, 0x72, 0x67, 0x25, 0x32, 0x30, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x25, 0x32, 0x30, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69,0x74, 0x79, 0x2E, 0x63, 0x72, 0x6C, 0x86, 0x43, 0x66, 0x69, 0x6C, 0x65, 0x3A, 0x2F, 0x2F, 0x5C, 0x5C, 0x6D, 0x61, 0x74, 0x68, 0x69, 0x6C, 0x64, 0x65, 0x5C, 0x43, 0x65, 0x72, 0x74, 0x45, 0x6E, 0x72, 0x6F, 0x6C, 0x6C, 0x5C, 0x4D, 0x65, 0x6E,0x74, 0x61, 0x6C, 0x69, 0x73, 0x2E, 0x6F, 0x72, 0x67, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x2E, 0x63, 0x72, 0x6C, 0x30, 0x81, 0xC6, 0x06, 0x08,0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x81, 0xB9, 0x30, 0x81, 0xB6, 0x30, 0x5A, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x4E, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x6D, 0x61, 0x74, 0x68, 0x69,0x6C, 0x64, 0x65, 0x2F, 0x43, 0x65, 0x72, 0x74, 0x45, 0x6E, 0x72, 0x6F, 0x6C, 0x6C, 0x2F, 0x6D, 0x61, 0x74, 0x68, 0x69, 0x6C, 0x64, 0x65, 0x5F, 0x4D, 0x65, 0x6E, 0x74, 0x61, 0x6C, 0x69, 0x73, 0x2E, 0x6F, 0x72, 0x67, 0x25, 0x32, 0x30, 0x43,0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x25, 0x32, 0x30, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x2E, 0x63, 0x72, 0x74, 0x30, 0x58, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x4C,0x66, 0x69, 0x6C, 0x65, 0x3A, 0x2F, 0x2F, 0x5C, 0x5C, 0x6D, 0x61, 0x74, 0x68, 0x69, 0x6C, 0x64, 0x65, 0x5C, 0x43, 0x65, 0x72, 0x74, 0x45, 0x6E, 0x72, 0x6F, 0x6C, 0x6C, 0x5C, 0x6D, 0x61, 0x74, 0x68, 0x69, 0x6C, 0x64, 0x65, 0x5F, 0x4D, 0x65,0x6E, 0x74, 0x61, 0x6C, 0x69, 0x73, 0x2E, 0x6F, 0x72, 0x67, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x2E, 0x63, 0x72, 0x74, 0x30, 0x0D, 0x06, 0x09,0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82, 0x02, 0x01, 0x00, 0x57, 0x1A, 0x4A,
			 0xEB, 0xDA, 0xE3, 0xCE, 0x4A, 0x8E, 0xE4, 0xF7, 0xE2, 0x81, 0x15, 0x9B, 0x90, 0x33, 0x1F, 0xEE, 0x5E, 0x47, 0x6D, 0xBD, 0xD4,0x73, 0xF6, 0xF0, 0xED, 0xA8, 0x62, 0x4B, 0x9E, 0xF8, 0x81, 0x48, 0x3F, 0x8C, 0xC1, 0x47, 0xC1, 0xDD, 0x10, 0xCE, 0x52, 0x10, 0x50, 0x04, 0x17, 0xBE, 0x00, 0x89, 0xF4, 0x8B, 0xD9, 0xC6, 0xDE, 0xE1, 0x70, 0xF6, 0xB0, 0xCA, 0xD1, 0xB8, 0x4F,0xC2, 0xAA, 0x35, 0xFF, 0x32, 0x81, 0xB9, 0x6A, 0xC7, 0x65, 0xB3, 0xD9, 0x1E, 0x80, 0xBD, 0xBC, 0x8F, 0x2A, 0x25, 0xA2, 0x81, 0x0A, 0xD5, 0x9E, 0xA8, 0x97, 0xF7, 0x0D, 0x30, 0x13, 0x2A, 0xEF, 0x4C, 0x12, 0xA7, 0x58, 0x5C, 0x9A, 0xFC, 0x00,0x89, 0xC6, 0x61, 0x5C, 0xBB, 0x26, 0x1B, 0xFB, 0x91, 0xF4, 0xD1, 0xED, 0x40, 0xA7, 0x76, 0x4B, 0x57, 0x7C, 0xDF, 0xEB, 0x7E, 0xDF, 0x81, 0xB2, 0x52, 0x9A, 0x6D, 0x79, 0x47, 0xBE, 0x74, 0x80, 0x5C, 0x9B, 0xF8, 0xDF, 0xA0, 0xE3, 0xE4, 0xE2,0xC2, 0x8A, 0xB9, 0xCB, 0xB2, 0xD6, 0x28, 0x16, 0xA1, 0x99, 0x15, 0x27, 0xC0, 0x2A, 0x9A, 0x07, 0xCA, 0xE1, 0x2D, 0x4F, 0x86, 0x13, 0x3F, 0x45, 0xAD, 0x42, 0xF5, 0x26, 0xF8, 0xEA, 0xD6, 0x83, 0xE4, 0xF7, 0xAC, 0x17, 0x10, 0xB5, 0xF8, 0xCC,0x0B, 0x4E, 0xB2, 0x2D, 0xCC, 0xF4, 0xD1, 0xF3, 0x33, 0x74, 0x40, 0x96, 0xD3, 0x6E, 0xC6, 0xB6, 0x11, 0x8E, 0x5E, 0x9B, 0x17, 0x20, 0x65, 0xBA, 0xC8, 0x74, 0xF3, 0xCD, 0x51, 0x46, 0xED, 0x1D, 0xB7, 0x87, 0x0D, 0x8F, 0x29, 0x8C, 0x97, 0x1C,0x37, 0xAF, 0xA6, 0x4B, 0xA6, 0x23, 0x26, 0xB2, 0x04, 0x17, 0xCC, 0x56, 0xF3, 0xE7, 0x1E, 0x23, 0x46, 0x6A, 0xDA, 0x44, 0xC3, 0xE1, 0x3F, 0x39, 0x7D, 0x75, 0x56, 0x53, 0x9D, 0x4B, 0x0C, 0x8B, 0x5B, 0x8D, 0xB4, 0xD2, 0xBF, 0x5F, 0xE4, 0xEA,0x33, 0xE5, 0xD8, 0x67, 0x60, 0x71, 0x1E, 0x87, 0xB7, 0xA9, 0x4A, 0x16, 0x72, 0x79, 0x05, 0xC7, 0x94, 0x83, 0xB7, 0xDF, 0x41, 0xCA, 0x41, 0xDA, 0xE5, 0xC5, 0xE5, 0x13, 0x51, 0x8C, 0x74, 0x2C, 0x4E, 0x64, 0x3A, 0x63, 0xD8, 0xB5, 0x9B, 0x6F,0xED, 0x23, 0x9F, 0x6E, 0x68, 0x32, 0x8A, 0xA0, 0x97, 0x50, 0xCD, 0x34, 0xEA, 0x6C, 0x17, 0x1E, 0x0D, 0xA7, 0x14, 0x23, 0x9A, 0x7E, 0x4F, 0x75, 0xAB, 0x6A, 0xAF, 0x03, 0xA2, 0x5C, 0x3E, 0xB2, 0x42, 0x36, 0xD4, 0x47, 0xF1, 0x9C, 0x4D, 0xA5,
			 0x78, 0x1A, 0x02, 0x63, 0x83, 0x55, 0x7E, 0x90, 0x73, 0xB5, 0xD6, 0xC9, 0x7F, 0xA8, 0x09, 0x44, 0x36, 0x53, 0x4A, 0x6B, 0x5B, 0x49, 0x80, 0x42, 0x8E, 0x18, 0x4B, 0x3B, 0xA9, 0x82, 0xAE, 0x4B, 0x3D, 0x99, 0x1C, 0xA6, 0x04, 0x3F, 0x01, 0x3C,0x05, 0x45, 0xBA, 0x36, 0x7E, 0x3B, 0x19, 0xCF, 0x83, 0x06, 0x9D, 0x78, 0x4A, 0x77, 0xA7, 0xCA, 0xA7, 0xFB, 0x21, 0x7E, 0xA1, 0x87, 0xAB, 0xE5, 0x0B, 0x99, 0x10, 0xA5, 0x79, 0x03, 0x61, 0xBA, 0xE7, 0x8F, 0xC8, 0x57, 0x11, 0x7F, 0x28, 0x70,0xF1, 0xA6, 0x57, 0x77, 0x8F, 0xDC, 0x98, 0x4C, 0x51, 0x8F, 0x8E, 0x0E, 0xA7, 0xE7, 0xD0, 0xB8, 0x2E, 0x6A, 0x46, 0x29, 0x83, 0x85, 0x59, 0x84, 0x87, 0xCB, 0xD8, 0xDE, 0x3D, 0xD7, 0x73, 0xE4, 0x52, 0xCB, 0xA2, 0xC2, 0x3D, 0x05, 0x5F, 0xB4,0xDA, 0x40, 0x75, 0x65, 0x72, 0x3E, 0x4C, 0x79, 0x0B, 0x9C, 0x2A, 0xBC, 0x93, 0x1F, 0x24, 0x28, 0x7A, 0x90, 0xA9, 0x7A, 0x0C, 0xCA, 0x53, 0x6B, 0xE5, 0x4E, 0x67, 0x6F, 0x5A, 0x13, 0xB2, 0x67, 0x5C, 0xE3, 0x36, 0x92, 0xD3, 0x6D, 0xF6, 0x69,0x87, 0x02, 0x84, 0x01, 0x59, 0x18, 0x96, 0xFB};
			int tests = 53;
			if (clone) {
				try {
					Certificate c2 = (Certificate)c.Clone();
					if (!c.Equals(c2))
						AddError("C-CSCBP1");
					tests += CheckServerCertBasicProps(c2, false);
				} catch {
					AddError("C-CSCBP2");
				}
			}
			try {
				DistinguishedName dn = c.GetDistinguishedName();
				StringDictionary sd = new StringDictionary();
				for(int i = 0; i < dn.Count; i++) {
					sd.Add(dn[i].ObjectID, dn[i].Value);
				}
				if (sd.Count != 7)
					AddError("C-CSCBP3");
				try {
					if (sd["2.5.4.3"] != "Mentalis.org Team") //name
						AddError("C-CSCBP3");
				} catch {
					AddError("C-CSCBP5");
				}
				try {
					if (sd["1.2.840.113549.1.9.1"] != "webmaster@mentalis.org") //email
						AddError("C-CSCBP6");
				} catch {
					AddError("C-CSCBP7");
				}
				try {
					if (sd["2.5.4.7"] != "Leuven") //city
						AddError("C-CSCBP8");
				} catch {
					AddError("C-CSCBP9");
				}
				try {
					if (sd["2.5.4.6"] != "BE") //country
						AddError("C-CSCBP10");
				} catch {
					AddError("C-CSCBP11");
				}
				try {
					if (sd["2.5.4.10"] != "The KPD-Team") //company
						AddError("C-CSCBP12");
				} catch {
					AddError("C-CSCBP13");
				}
				try {
					if (sd["2.5.4.11"] != "Software Development") //department
						AddError("C-CSCBP14");
				} catch {
					AddError("C-CSCBP15");
				}
				try {
					if (sd["2.5.4.8"] != "Brabant") //state
						AddError("C-CSCBP16");
				} catch {
					AddError("C-CSCBP17");
				}
			} catch {
				AddError("C-CSCBP18");
			}
			try {
				if (!ArrayEquals(c.GetCertHash(HashType.SHA1), SHA1Hash))
					AddError("C-CSCBP19");
			} catch {
				AddError("C-CSCBP20");
			}
			try {
				if (!ArrayEquals(c.GetCertHash(HashType.MD5), MD5Hash))
					AddError("C-CSCBP21");
			} catch {
				AddError("C-CSCBP22");
			}
			try {
				if (c.GetEffectiveDate().ToFileTime() != 0x1C29FBC26000A00)
					AddError("C-CSCBP23");
			} catch {
				AddError("C-CSCBP24");
			}
			try {
				if (c.GetExpirationDate().ToFileTime() != 0x1C3BE8F04678600)
					AddError("C-CSCBP25");
			} catch {
				AddError("C-CSCBP26");
			}
			try {
				StringCollection sc = c.GetEnhancedKeyUsage();
				if (sc.Count != 1)
					AddError("C-CSCBP27");
				if (!sc.Contains("1.3.6.1.5.5.7.3.1"))
					AddError("C-CSCBP28");
			} catch {
				AddError("C-CSCBP29");
			}
			try {
				if (c.GetFormat() != "X509")
					AddError("C-CSCBP30");
			} catch {
				AddError("C-CSCBP31");
			}
			try {
				if (!ArrayEquals(BitConverter.GetBytes(c.GetHashCode()), HashCode))
					AddError("C-CSCBP32");
			} catch {
				AddError("C-CSCBP33");
			}
			try {
				if (c.GetIntendedKeyUsage() != 48) // 48 == Data Encipherment and Key Encipherment
					AddError("C-CSCBP34");
			} catch {
				AddError("C-CSCBP35");
			}
			try {
				if (c.GetIssuerName() != "Mentalis.org Certificate Authority")
					AddError("C-CSCBP36");
			} catch {
				AddError("C-CSCBP37");
			}
			try {
				if (c.GetKeyAlgorithm() != "1.2.840.113549.1.1.5")
					AddError("C-CSCBP38");
			} catch {
				AddError("C-CSCBP39");
			}
			try {
				if (c.GetKeyAlgorithmParametersString() != "0500")
					AddError("C-CSCBP40");
			} catch {
				AddError("C-CSCBP41");
			}
			try {
				if (c.GetName() != "Mentalis.org Team")
					AddError("C-CSCBP42");
			} catch {
				AddError("C-CSCBP43");
			}
			try {
				if (c.GetPublicKeyLength() != 1024)
					AddError("C-CSCBP44");
			} catch {
				AddError("C-CSCBP45");
			}
			try {
				if (c.GetPublicKeyString() != "30818902818100BC7B73D8F2A48B17684E49A1425252FD1D8354E2B3202B79BE5764EF2ADD6D413AE8F6E327FBFC4FB3F274357FEE7478BB28AE58975F73BE510F49E4545665B3B3443FD52C27A0DF6B6D8A3AAB5A42C3DEC14CB9A85955C57A99661FB99FE08363E460BEFFFF33C4B6F9F87B12FD6F5D7F2A8AEF6EFA2DA5FAF7EBD72B8B87170203010001")
					AddError("C-CSCBP46");
			} catch {
				AddError("C-CSCBP47");
			}
			try {
				if (!ArrayEquals(RawData, c.GetRawCertData()))
					AddError("C-CSCBP48");
			} catch {
				AddError("C-CSCBP49");
			}
			try {
				if (!ArrayEquals(c.GetSerialNumber(), Serial))
					AddError("C-CSCBP50");
			} catch {
				AddError("C-CSCBP51");
			}
			try {
				if (!ArrayEquals(RawData, c.ToCerBuffer()))
					AddError("C-CSCBP52");
			} catch {
				AddError("C-CSCBP53");
			}
			return tests;
		}
	}
}
