using System;
using System.Collections;
using System.Diagnostics;
using System.IO;
using Selenium;
using NUnit.Framework;
using Orciid.Core;

namespace webtests
{
	[TestFixture]
	[Category("UsesSelenium")]
	public class TestFilterResults: SeleniumTestFixture
	{
		private Utilities util;
		private ArrayList collections = new ArrayList();

		private string CreateTempDirectory()
		{
			string path = Path.GetTempFileName();
			File.Delete(path);
			Directory.CreateDirectory(path);
			Debug.WriteLine("Created temporary directory " + path);
			return path;
		}

		private Collection GetTempCollection(string title)
		{
			Collection coll = new Collection();
			coll.Title = title;
			coll.Description = coll.Title;
			coll.ResourcePath = CreateTempDirectory();
			coll.Update();
			collections.Add(coll);
			return coll;
		}

		protected override void MoreSetUp()
		{
			// current directory should be webtests/bin/Debug
			string dsdir = Path.Combine(Directory.GetCurrentDirectory(), @"..\..\datasets");

			util = new Utilities(dsdir);

			User.GetByLogin("admin", "admin").Activate(null);
		}

		protected override void MoreTearDown()
		{			
			foreach (Collection c in collections)
				if (c != null)
				{
					Directory.Delete(c.ResourcePath, true);
					c.Delete();
				}
		}

		const string FORMAT_STRING = "TestFilterResultPainting";

		private void AddImage(Collection coll, string title, bool personal)
		{
			Image image = coll.CreateImage(personal);
			image["Title"].Add(title);
			image["Format"].Add(FORMAT_STRING);
			image.Update();
		}

		[Test]
		public void TestFilterResult() 
		{
			Collection coll = GetTempCollection("TestFilterResult");
			coll.CreateDublinCoreFields();

			// clear any saved last searches to start with a clean search form
			Properties props = Properties.GetProperties(User.CurrentUser());
			if (props != null)
				props.Remove("lastsearch");

			AddImage(coll, "Picasso", false);
			AddImage(coll, "Monet", false);
			AddImage(coll, "Pollock", false);
			AddImage(coll, "Webb", true);
			AddImage(coll, "Dali", true);
			
			new FullTextIndex(coll).UpdateIndex();

			LoginAdministrator(selenium);

			// filter regular collections
			selenium.Click("link=Search");
			selenium.WaitForPageToLoad("30000");
			Assert.AreEqual("Search Collections", selenium.GetText("//h1[@id='Heading']"));
			Assert.IsTrue(selenium.IsTextPresent("TestFilterResult"));
			selenium.Click("CollectionSelection:G0C" + coll.ID.ToString());
			selenium.WaitForPageToLoad("30000");
			selenium.Type("KeywordField", FORMAT_STRING);
			selenium.Click("SearchButton");
			selenium.WaitForPageToLoad("30000");
			Assert.IsTrue(selenium.IsTextPresent("Page 1 of 1 (3 records found)"));
			selenium.Type("FilterTextBox", "Picasso");
			selenium.Click("FilterButton");
			selenium.WaitForPageToLoad("30000");
			Assert.IsFalse(selenium.IsTextPresent("No matching image records were found."));
			Assert.IsTrue(selenium.IsTextPresent("Page 1 of 1 (1 records found)"));

			// filter personal images
			selenium.Click("link=My Images");
			selenium.WaitForPageToLoad("30000");
			Assert.AreEqual("My Images", selenium.GetText("//h1[@id='Heading']"));
			selenium.Click("link=View your images");
			selenium.WaitForPageToLoad("30000");
			Assert.AreEqual("Search Results", selenium.GetText("//h1[@id='Title']"));
			Assert.IsTrue(selenium.IsTextPresent("Page 1 of"));
			selenium.Type("FilterTextBox", FORMAT_STRING);
			selenium.Click("FilterButton");
			selenium.WaitForPageToLoad("30000");
			Assert.IsFalse(selenium.IsTextPresent("No matching image records were found."));
			Assert.IsTrue(selenium.IsTextPresent("Page 1 of 1 (2 records found)"));
		}
	}
}
