using System;
using System.Collections;
using System.Diagnostics;
using System.IO;
using System.Data;
using System.Data.SqlClient;
using NUnit.Framework;
using Orciid.Core;

namespace webtests
{
	[TestFixture]
	public class TestRemoteCollection
	{
		private ArrayList collections = new ArrayList();
		private Utilities util;

		[SetUp]
		protected void SetUp()
		{
			// current directory should be webtests/bin/Debug
			string dsdir = Path.Combine(Directory.GetCurrentDirectory(), @"..\..\datasets");

			util = new Utilities(dsdir);

			User.GetByLogin("admin", "admin").Activate(null);
		}

		[TearDown]
		protected void TearDown()
		{
			foreach (Collection c in collections)
				if (c != null)
				{
					if (!(c is RemoteCollection))
						Directory.Delete(c.ResourcePath, true);
					c.Delete();
				}
		}

		private string CreateTempDirectory()
		{
			// need to create temp directory in public location
			// so that both this process and the web server process can
			// access it
			string rpath = Configuration.Instance.GetString("content.resourcepath");
			Assert.IsTrue(Directory.Exists(rpath));
			int i = 0;
			string path;
			do
			{
				path = Path.Combine(rpath, "temp" + i.ToString());
				i++;
			} while (Directory.Exists(path));
			Directory.CreateDirectory(path);
			Debug.WriteLine("Created temporary directory " + path);
			return path;
		}

		private Collection GetTempCollection()
		{
			Collection coll = new Collection();
			coll.Title = "__Temp Collection__";
			coll.Description = coll.Title;
			coll.ResourcePath = CreateTempDirectory();
			coll.Update();
			collections.Add(coll);
			return coll;
		}

		private string GetRemoteUrl()
		{
			UriBuilder uri = new UriBuilder(Configuration.Instance.GetString("selenium.baseurl"));
			uri.UserName = "admin";
			uri.Password = "admin";
			return uri.ToString();
		}

		[Test]
		public void TestRemoteCollectionSearch()
		{
			Collection coll = GetTempCollection();
			coll.CreateDublinCoreFields();
			CollectionShareEntry se = coll.CreateShareEntry();
			se.User = "admin";
			se.Subnet = "0.0.0.0";
			se.Mask = "0.0.0.0";
			se.Update();

			RemoteCollection rcoll = new RemoteCollection();
			rcoll.Title = "temp_" + Guid.NewGuid().ToString();
			rcoll.Description = rcoll.Title;
			rcoll.ResourcePath = @"\\invalid\resourcepath";
			rcoll.RemoteUrl = GetRemoteUrl() + "," + coll.ID.ToString();
			rcoll.Update();
			collections.Add(rcoll);

			Image image = coll.CreateImage(false);
			image["Title"].Add("Testing");
			image["Subject"].Add("Special char: <\u000B>");
			image.Update();

			new FullTextIndex(coll).UpdateIndex();

			Search.SearchParameters p = new Orciid.Core.Search.SearchParameters();
			p.collections.Add(rcoll);
			ExactTextCondition tc = new ExactTextCondition(true);
			SearchField sf = new SearchField();
			sf.Name = "Title";
			tc.Field = sf;
			tc.Keyword = "Testing";
			p.conditions.Add(tc);

			ImageIdentifier[] ids = Orciid.Core.Search.Execute(p, 10, false);

			Assert.AreEqual(1, ids.Length);
			Assert.AreEqual(rcoll.ID, ids[0].CollectionID);

			Image result = Image.GetByID(ids[0]);
			Assert.AreEqual("Testing", result["Title"][0]);		
			Assert.AreEqual("Special char: <>", result["Subject"][0]);
		}
	}
}
