using System;
using System.Collections;
using System.Diagnostics;

// Creation date: 22.04.2002
// Checked: 17.05.2002
// Author: Otto Mayer, mot@root.ch
// Version 0.02.00

// copyright (C) 2002 root-software ag  -  Brglen Switzerland  -  www.root.ch; Otto Mayer, Stefan Spirig, Roger Gartenmann
// This library is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation, version 2.1 of the License.
// This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details. You
// should have received a copy of the GNU Lesser General Public License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA www.opensource.org/licenses/lgpl-license.html

namespace Root.Reports {
  /// <summary>Container for report objects</summary>
  public abstract class Container	: RepObj, IEnumerable {
    /// <summary>Pointer to the first report object of the container</summary>
    internal RepObj repObj_First;

    /// <summary>Pointer to the last report object of the container</summary>
    internal RepObj repObj_Last;

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Creates a new container.</summary>
    protected Container() {
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Creates a new container.</summary>
    public Container(Double rWidth, Double rHeight) {
      this.rWidth = rWidth;
      this.rHeight = rHeight;
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Appends a report object to the container.</summary>
    /// <param name="repObj">Report object that must be added to the container</param>
    internal void Add(RepObj repObj) {
      Add(repObj, null);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    internal void Add(Double rX, Double rY, RepObj repObj) {
      repObj.matrixD.rDX = rX;
      repObj.matrixD.rDY = rY;
      Add(repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container.</summary>
    /// <param name="repObj">Report object that must be added to the container</param>
    /// <param name="repObj_Pos">The new report object will be inserted before the specified position.
    /// If the position is null it will be appended to the end of the list.</param>
    internal void Add(RepObj repObj, RepObj repObj_Pos) {
      if (repObj.container != null) {
        throw new ReportException("Report objects cannot be added to more than one container");
      }
      Debug.Assert(repObj.repObj_Next == null);
      Debug.Assert(repObj.repObj_Prev == null);

      if (repObj_First == null) {
        Debug.Assert(repObj_Last == null);
        repObj_First = repObj_Last = repObj;
      }
      else {
        if (repObj_Pos == null) {
          repObj.repObj_Prev = repObj_Last;
          repObj_Last.repObj_Next = repObj;
          repObj_Last = repObj;
        }
        else {
          if (repObj_Pos.container != this) {
            throw new ReportException("The report object indicating the position is not a member of the container");
          }
          repObj.repObj_Next = repObj_Pos;
          if (repObj_Pos.repObj_Prev == null) {
            Debug.Assert(repObj_First == repObj_Pos);
            repObj_First = repObj;
          }
          else {
            repObj_Pos.repObj_Prev.repObj_Next = repObj;
          }

          repObj.repObj_Prev = repObj_Pos.repObj_Prev;
          repObj_Pos.repObj_Prev = repObj;
        }
      }
      repObj.container = this;
      repObj.OnAdded();
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rAlignH">Horizontal alignment of the report object relative to [X].</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="rAlignV">Vertical alignment of the report object relative to [Y].</param>
    /// <param name="repObj">Report object to add to the container</param>
    /// <param name="repObj_Pos">The new report object will be inserted before the specified position.
    /// If the position is null it will be appended to the end of the list.</param>
    internal void AddAligned(Double rX, Double rAlignH, Double rY, Double rAlignV, RepObj repObj, RepObj repObj_Pos) {
      repObj.matrixD.rDX = rX;
      repObj.rAlignH = rAlignH;
      repObj.matrixD.rDY = rY;
      repObj.rAlignV = rAlignV;
      Add(repObj, repObj_Pos);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rAlignH">Horizontal alignment of the report object relative to [X].</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="rAlignV">Vertical alignment of the report object relative to [Y].</param>
    /// <param name="repObj">Report object to add to the container</param>
    internal void AddAligned(Double rX, Double rAlignH, Double rY, Double rAlignV, RepObj repObj) {
      AddAligned(rX, rAlignH, rY, rAlignV, repObj, null);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    /// <param name="repObj_Pos">The new report object will be inserted before the specified position.
    /// If the position is null it will be appended to the end of the list.</param>
    public void AddLT(Double rX, Double rY, RepObj repObj, RepObj repObj_Pos) {
      AddAligned(rX, RepObj.rAlignLeft, rY, RepObj.rAlignTop, repObj, repObj_Pos);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    internal void AddLT(Double rX, Double rY, RepObj repObj) {
      AddLT(rX, rY, repObj, null);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    public void AddLT_MM(Double rX, Double rY, RepObj repObj) {
      AddLT(RT.rPointFromMM(rX), RT.rPointFromMM(rY), repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    internal void AddCT(Double rX, Double rY, RepObj repObj) {
      AddAligned(rX, RepObj.rAlignCenter, rY, RepObj.rAlignTop, repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    public void AddCT_MM(Double rX, Double rY, RepObj repObj) {
      AddCT(RT.rPointFromMM(rX), RT.rPointFromMM(rY), repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    internal void AddCC(Double rX, Double rY, RepObj repObj) {
      AddAligned(rX, RepObj.rAlignCenter, rY, RepObj.rAlignCenter, repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    public void AddCC_MM(Double rX, Double rY, RepObj repObj) {
      AddCC(RT.rPointFromMM(rX), RT.rPointFromMM(rY), repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    internal void AddCB(Double rX, Double rY, RepObj repObj) {
      AddAligned(rX, RepObj.rAlignCenter, rY, RepObj.rAlignBottom, repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    public void AddCB_MM(Double rX, Double rY, RepObj repObj) {
      AddCB(RT.rPointFromMM(rX), RT.rPointFromMM(rY), repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    internal void AddRT(Double rX, Double rY, RepObj repObj) {
      AddAligned(rX, RepObj.rAlignRight, rY, RepObj.rAlignTop, repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    public void AddRT_MM(Double rX, Double rY, RepObj repObj) {
      AddRT(RT.rPointFromMM(rX), RT.rPointFromMM(rY), repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    internal void AddRC(Double rX, Double rY, RepObj repObj) {
      AddAligned(rX, RepObj.rAlignRight, rY, RepObj.rAlignCenter, repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Adds a report object to the container and sets the alignment.</summary>
    /// <param name="rX">X-coordinate of the report object</param>
    /// <param name="rY">Y-coordinate of the report object</param>
    /// <param name="repObj">Report object to add to the container</param>
    public void AddRC_MM(Double rX, Double rY, RepObj repObj) {
      AddRC(RT.rPointFromMM(rX), RT.rPointFromMM(rY), repObj);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Returns an enumerator that can iterate through the report objects.</summary>
    /// <returns>An enumerator that can be used to iterate through the report objects</returns>
    public IEnumerator GetEnumerator() {
      return new RepObjEnumerator(this);
    }

    //****************************************************************************************************
    /// <summary>Enumerator class to iterate through the report objects.</summary>
    public class RepObjEnumerator: IEnumerator {
      /// <summary>Container</summary>
      Container container;

      /// <summary>Current report object</summary>
      RepObj repObj_Cur;

      //----------------------------------------------------------------------------------------------------x
      /// <summary>Creates the report object enumerator</summary>
      /// <param name="container">The report objects of this container will be enumerated</param>
      public RepObjEnumerator(Container container) {
        this.container = container;
      }

      //----------------------------------------------------------------------------------------------------x
      /// <summary>Sets the enumerator to its initial position, which is before the first element in the collection.</summary>
      public void Reset() {
        repObj_Cur = null;
      }

      //----------------------------------------------------------------------------------------------------x
      /// <summary>Advances the enumerator to the next report object of the container.</summary>
      /// <returns>true if the enumerator was successfully advanced to the next report object,
      /// false if the enumerator has passed the end of the collection.</returns>
      public bool MoveNext() {
        if (repObj_Cur == null) {
          repObj_Cur = container.repObj_First;
        }
        else {
          repObj_Cur = repObj_Cur.repObj_Next;
        }
        return (repObj_Cur != null);
      }

      //----------------------------------------------------------------------------------------------------x
      /// <summary>Gets the current report object of the container.</summary>
      object IEnumerator.Current {
        get { return repObj_Cur; }
      }
    }

  }
}
