using System;
using System.Collections;
using System.IO;
using System.Drawing;
using System.Drawing.Imaging;
using System.Drawing.Drawing2D;
using NUnit.Framework;
using System.Diagnostics;

namespace Orciid.Media.Converters
{
	[TestFixture]
	public class TestBitmapConverter
	{
		private static Bitmap defaultbitmap;
		private static ArrayList tempfiles = new ArrayList();

		static TestBitmapConverter()
		{
			defaultbitmap = new Bitmap(128, 128, PixelFormat.Format24bppRgb);
			for (int x = 0; x < 128; x++)
				for (int y = 0; y < 128; y++)
					defaultbitmap.SetPixel(x, y, Color.FromArgb(x * 2, y * 2, x + y));
		}

		private static Bitmap GetTestBitmap()
		{
			return defaultbitmap.Clone() as Bitmap;
		}

		private static string GetTempFileName(string ext)
		{
			string target = Util.Tools.GetTempFileName(ext);
			tempfiles.Add(target);
			Debug.WriteLine("Created temporary file " + target);
			return target;
		}

		[SetUp]
		public void Setup()
		{
			Debug.Listeners.Add(new TextWriterTraceListener(Console.Out));
			Debug.AutoFlush = true;
		}

		[TearDown]
		public void Cleanup()
		{
			Debug.Listeners.Clear();
			while (tempfiles.Count > 0)
			{
				try
				{
					File.Delete(tempfiles[0] as string);
				}
				finally
				{
					tempfiles.RemoveAt(0);
				}
			}
		}


		[Test]
		public void TestPngToBitmapConversion()
		{
			string tempfile = GetTempFileName("png");
			string testfile = GetTempFileName("bmp");
			using (Bitmap original = GetTestBitmap())
			{
				original.Save(tempfile, ImageFormat.Png);
				Assert.IsTrue(MediaConverter.Convert(tempfile, testfile));
				Assert.IsTrue(File.Exists(testfile));
				using (Bitmap test = new Bitmap(testfile))
				{
					Assert.AreEqual(original.Width, test.Width);
					Assert.AreEqual(original.Height, test.Height);
				}
			}
		}

		[Test]
		public void TestPngToTiffConversion()
		{
			string tempfile = GetTempFileName("png");
			string testfile = GetTempFileName("tif");
			using (Bitmap original = GetTestBitmap())
			{
				original.Save(tempfile, ImageFormat.Png);
				Assert.IsTrue(MediaConverter.Convert(tempfile, testfile));
				Assert.IsTrue(File.Exists(testfile));
				using (Bitmap test = new Bitmap(testfile))
				{
					Assert.AreEqual(original.Width, test.Width);
					Assert.AreEqual(original.Height, test.Height);
				}
			}
		}

		[Test]
		public void TestPngToJpegConversion()
		{
			string tempfile = GetTempFileName("png");
			string testfile = GetTempFileName("jpg");
			using (Bitmap original = GetTestBitmap())
			{
				original.Save(tempfile, ImageFormat.Png);
				Assert.IsTrue(MediaConverter.Convert(tempfile, testfile));
				Assert.IsTrue(File.Exists(testfile));
				using (Bitmap test = new Bitmap(testfile))
				{
					Assert.AreEqual(original.Width, test.Width);
					Assert.AreEqual(original.Height, test.Height);
				}
			}
		}

		[Test]
		public void TestResizeConversion()
		{
			string tempfile = GetTempFileName("png");
			string testfile = GetTempFileName("bmp");
			using (Bitmap original = GetTestBitmap())
				original.Save(tempfile, ImageFormat.Png);
			Parameters parameters = new Parameters(50, 50, Parameters.Unspecified);
			Assert.IsTrue(MediaConverter.Convert(tempfile, testfile, parameters));
			Assert.IsTrue(File.Exists(testfile));
			using (Bitmap test = new Bitmap(testfile))
			{
				Assert.AreEqual(50, test.Width);
				Assert.AreEqual(50, test.Height);
			}
		}

		[Test]
		public void TestAddBorderConversion()
		{
			string tempfile = GetTempFileName("png");
			string testfile = GetTempFileName("bmp");
			using (Bitmap original = GetTestBitmap())
				original.Save(tempfile, ImageFormat.Png);
			Color color = Color.FromArgb(255, 123, 123, 123);
			Parameters parameters = new Parameters(50, 150, true, color, Parameters.Unspecified);
			Assert.IsTrue(MediaConverter.Convert(tempfile, testfile, parameters));
			Assert.IsTrue(File.Exists(testfile));
			using (Bitmap test = new Bitmap(testfile))
			{
				Assert.AreEqual(50, test.Width);
				Assert.AreEqual(150, test.Height);
				Assert.AreEqual(color, test.GetPixel(0, 0));
				Assert.AreEqual(color, test.GetPixel(49, 49));
				Assert.AreEqual(color, test.GetPixel(0, 100));
				Assert.AreEqual(color, test.GetPixel(49, 149));
				Assert.IsFalse(color == test.GetPixel(49, 50));
				Assert.IsFalse(color == test.GetPixel(0, 99));
			}
		}

		[Test]
		public void TestAddBorderToLargeCanvasConversion()
		{
			string tempfile = GetTempFileName("png");
			string testfile = GetTempFileName("bmp");
			using (Bitmap original = GetTestBitmap())
				original.Save(tempfile, ImageFormat.Png);
			Color color = Color.FromArgb(255, 123, 123, 123);
			Parameters parameters = new Parameters(256, 512, true, color, Parameters.Unspecified);
			Assert.IsTrue(MediaConverter.Convert(tempfile, testfile, parameters));
			Assert.IsTrue(File.Exists(testfile));
			using (Bitmap test = new Bitmap(testfile))
			{
				Assert.AreEqual(256, test.Width);
				Assert.AreEqual(512, test.Height);
				Assert.AreEqual(color, test.GetPixel(0, 0));
				Assert.AreEqual(color, test.GetPixel(0, 511));
				Assert.AreEqual(color, test.GetPixel(255, 0));
				Assert.AreEqual(color, test.GetPixel(255, 511));
			}
		}
	}
}
