using System;
using System.Collections;
using System.IO;
using System.Data;
using System.Data.SqlClient;
using NUnit.Framework;
using Orciid.Core;

namespace webtests
{
	[TestFixture]
	public class TestImport
	{
		private ArrayList collections = new ArrayList();
		private Utilities util;

		private Collection coll;

		[SetUp]
		protected void SetUp()
		{
			// current directory should be webtests/bin/Debug
			string dsdir = Path.Combine(Directory.GetCurrentDirectory(), @"..\..\datasets");

			util = new Utilities(dsdir);

			User.GetByLogin("admin", "admin").Activate(null);

			coll = GetTempCollection();
			coll.CreateDublinCoreFields();
		}

		[TearDown]
		protected void TearDown()
		{
			foreach (Collection c in collections)
				if (c != null)
					c.Delete();
		}

		private Collection GetTempCollection()
		{
			Collection coll = new Collection();
			coll.Title = "temp_" + Guid.NewGuid().ToString();
			coll.Description = coll.Title;
			coll.ResourcePath = @"\\invalid\resourcepath";
			coll.Update();
			collections.Add(coll);
			return coll;
		}

		[Test]
		public void TestSimpleImport()
		{
			string data = 
@"Identifier,Resource,Title
A001,A001.jpg,Testing
";

			ImportFileReader r = new ImportFileReader(
				new StringReader(data), ImportFileReader.FileType.CSV);
			DataTable table = r.GetData();

			Import import = new Import(coll,
				table,
				r.GetMapping(coll),
				"Identifier",
				"Resource",
				false,
				false,
				DateInterpreter.Format.YearOnly);

			Import.ImportStatus status = import.Run();

			Console.WriteLine(status.Errors);

			Assert.AreEqual(1, status.Read);
			Assert.AreEqual(1, status.Added);
			Assert.AreEqual(0, status.Replaced);
			Assert.AreEqual("", status.Errors);

			ImageIdentifier[] ids = 
				(ImageIdentifier[])status.ImageIdentifiers.ToArray(typeof(ImageIdentifier));

			Assert.AreEqual(1, ids.Length);
			Assert.AreEqual(coll.ID, ids[0].CollectionID);

			Image result = Image.GetByID(ids[0]);
			Assert.AreEqual("Testing", result["Title"][0]);
			Assert.AreEqual("A001", result["Identifier"][0]);
			Assert.AreEqual("A001.jpg", result.Resource);
		}

		[Test]
		public void TestSimpleDateImport()
		{
			string data = 
				@"Identifier,Resource,Title,Date
B001,B001.jpg,Testing,2006
";
			ImportFileReader r = new ImportFileReader(
				new StringReader(data), ImportFileReader.FileType.CSV);
			DataTable table = r.GetData();

			Import import = new Import(coll,
				table,
				r.GetMapping(coll),
				"Identifier",
				"Resource",
				false,
				false,
				DateInterpreter.Format.YearOnly);

			Import.ImportStatus status = import.Run();

			Console.WriteLine(status.Errors);

			Assert.AreEqual(1, status.Read);
			Assert.AreEqual(1, status.Added);
			Assert.AreEqual(0, status.Replaced);
			Assert.AreEqual("", status.Errors);

			ImageIdentifier[] ids = 
				(ImageIdentifier[])status.ImageIdentifiers.ToArray(typeof(ImageIdentifier));

			Assert.AreEqual(1, ids.Length);
			Assert.AreEqual(coll.ID, ids[0].CollectionID);

			Image result = Image.GetByID(ids[0]);
			Assert.AreEqual("Testing", result["Title"][0]);
			Assert.AreEqual("B001", result["Identifier"][0]);
			Assert.AreEqual("B001.jpg", result.Resource);
			
			Assert.AreEqual("2006", result["Date"][0]);
			FieldDataDate date = (FieldDataDate)result["Date"];
			Assert.AreEqual(2006, date.GetStartDate(0));
			Assert.AreEqual(2006, date.GetEndDate(0));
			Assert.AreEqual("2006", date.GetOriginalValue(0));
		}

		[Test]
		public void TestComplexDateImport()
		{
			string data = 
				@"Identifier,Resource,Title,StartDate,EndDate,OriginalDate,Date
B001,B001.jpg,Testing,1997,2006,,
B002,B002.jpg,More testing,,,1870-1872,
B003,B003.jpg,Yet more testing,,,1871-1873,early 1870s
B004,B004.jpg,And more testing,1872,1877,1874-1876,ca. 1875
";
			ImportFileReader r = new ImportFileReader(
				new StringReader(data), ImportFileReader.FileType.CSV);
			DataTable table = r.GetData();

			Assert.AreEqual(4, table.Rows.Count);

			FieldMapping mapping = r.GetMapping(coll);
			mapping.AddMapping("StartDate", "Date::start");
			mapping.AddMapping("EndDate", "Date::end");
			mapping.AddMapping("OriginalDate", "Date::internal");

			Import import = new Import(coll,
				table,
				mapping,
				"Identifier",
				"Resource",
				false,
				false,
				DateInterpreter.Format.YearOnly);

			Import.ImportStatus status = import.Run();

			Console.WriteLine(status.Errors);

			Assert.AreEqual(4, status.Read);
			Assert.AreEqual(4, status.Added);
			Assert.AreEqual(0, status.Replaced);
			Assert.AreEqual("", status.Errors);

			ImageIdentifier[] ids = 
				(ImageIdentifier[])status.ImageIdentifiers.ToArray(typeof(ImageIdentifier));

			Assert.AreEqual(4, ids.Length);
			Assert.AreEqual(coll.ID, ids[0].CollectionID);

			Image result = Image.GetByID(ids[0]);
			Assert.AreEqual("Testing", result["Title"][0]);
			Assert.AreEqual("B001", result["Identifier"][0]);
			Assert.AreEqual("B001.jpg", result.Resource);			
			Assert.AreEqual("1997-2006", result["Date"][0]);
			FieldDataDate date = (FieldDataDate)result["Date"];
			Assert.AreEqual(1997, date.GetStartDate(0));
			Assert.AreEqual(2006, date.GetEndDate(0));
			Assert.IsNull(date.GetOriginalValue(0));

			result = Image.GetByID(ids[1]);
			Assert.AreEqual("More testing", result["Title"][0]);
			Assert.AreEqual("B002", result["Identifier"][0]);
			Assert.AreEqual("B002.jpg", result.Resource);			
			Assert.AreEqual("1870-1872", result["Date"][0]);
			date = (FieldDataDate)result["Date"];
			Assert.AreEqual(1870, date.GetStartDate(0));
			Assert.AreEqual(1872, date.GetEndDate(0));
			Assert.AreEqual("1870-1872", date.GetOriginalValue(0));

			result = Image.GetByID(ids[2]);
			Assert.AreEqual("Yet more testing", result["Title"][0]);
			Assert.AreEqual("B003", result["Identifier"][0]);
			Assert.AreEqual("B003.jpg", result.Resource);			
			Assert.AreEqual("early 1870s", result["Date"][0]);
			date = (FieldDataDate)result["Date"];
			Assert.AreEqual(1871, date.GetStartDate(0));
			Assert.AreEqual(1873, date.GetEndDate(0));
			Assert.AreEqual("1871-1873", date.GetOriginalValue(0));
		
			result = Image.GetByID(ids[3]);
			Assert.AreEqual("And more testing", result["Title"][0]);
			Assert.AreEqual("B004", result["Identifier"][0]);
			Assert.AreEqual("B004.jpg", result.Resource);			
			Assert.AreEqual("ca. 1875", result["Date"][0]);
			date = (FieldDataDate)result["Date"];
			Assert.AreEqual(1872, date.GetStartDate(0));
			Assert.AreEqual(1877, date.GetEndDate(0));
			Assert.AreEqual("1874-1876", date.GetOriginalValue(0));
		}

	}
}
