using System;
using System.Data;
using System.Collections;

namespace Orciid.Core
{
	/// <summary>
	/// Announcement
	/// </summary>
	/// <remarks>
	/// This class holds system announcement information
	/// </remarks>
	public class Announcement
	{
#if DEMO
#warning Compiling for Demo site
		public int id;
		public int userid;
		public DateTime posted;
		public string title;
		public string text;
		public string linkurl;
		public string linktitle;

		public Announcement()
		{
		}
#else
		private int id;
		private int userid;
		private DateTime posted;
		private string title;
		private string text;
		private string linkurl;
		private string linktitle;

		private Announcement()
		{
		}
#endif

		/// <summary>
		/// Retrieve announcements
		/// </summary>
		/// <remarks>
		/// This method creates and returns an array of all announcements in the system
		/// in reverse chronological order
		/// </remarks>
		/// <returns>A sorted array of announcements</returns>
		public static Announcement[] GetAnnouncements()
		{
			using (DBConnection conn = DBConnector.GetConnection())
			{
				Query query = new Query(conn,
					@"SELECT ID,UserID,Posted,Title,Text,Link,LinkTitle 
					FROM Announcements ORDER BY Posted DESC");
				DataTable table = conn.SelectQuery(query);
				ArrayList list = new ArrayList();
				foreach (DataRow row in table.Rows)
				{
					Announcement a = new Announcement();
					a.id = conn.DataToInt(row["ID"], 0);
					a.userid = conn.DataToInt(row["UserID"], 0);
					a.posted = conn.DataToDateTime(row["Posted"]);
					a.title = conn.DataToString(row["Title"]);
					a.text = conn.DataToString(row["Text"]);
					a.linkurl = conn.DataToString(row["Link"]);
					a.linktitle = conn.DataToString(row["LinkTitle"]);
					list.Add(a);
				}
				return (Announcement[])list.ToArray(typeof(Announcement));
			}
		}

        /// <summary>
        /// Get date of latest announcement
        /// </summary>
        /// <remarks>
        /// This method returns the date when the latest/newest announcement was posted
        /// </remarks>
        /// <returns>the date when the latest/newest announcement was posted</returns>
		public static DateTime GetLastAnnouncementDateTime()
		{
			using (DBConnection conn = DBConnector.GetConnection())
			{
				Query query = new Query(conn,
					@"SELECT MAX(Posted) AS Latest FROM Announcements");
				DateTime result = conn.DataToDateTime(conn.ExecScalar(query), DateTime.MinValue);
				return result;
			}
		}

		/// <summary>
		/// Create announcement
		/// </summary>
		/// <remarks>
		/// Adds an announcement to the system
		/// </remarks>
		/// <param name="title">The title of the announcement</param>
		/// <param name="text">The text of the announcement</param>
		/// <param name="link">Optional hyperlink to be displayed</param>
		/// <param name="linktitle">Title for optional hyperlink</param>
		public static void AddAnnouncement(string title, string text, string link, string linktitle)
		{
			User.RequirePrivilege(Privilege.ManageAnnouncements);
			if (title == null || text == null)
				throw new CoreException("Title and text must be specified for announcement");
			User user = User.CurrentUser();
			if (title.Length > 255)
				title = title.Substring(0, 255);
			if (link != null && link.Length > 255)
				link = link.Substring(0, 255);
			if (linktitle != null && linktitle.Length > 255)
				linktitle = linktitle.Substring(0, 255);
			using (DBConnection conn = DBConnector.GetConnection())
			{
				Query query = new Query(conn, 
					@"INSERT INTO Announcements (UserID,Posted,Title,Text,Link,LinkTitle) 
					VALUES ({userid},{posted},{title},{text},{link},{linktitle})");
				query.AddParam("userid", user.ID);
				query.AddParam("posted", DateTime.Now);
				query.AddParam("title", title);
				query.AddParam("text", text);
				query.AddParam("link", link);
				query.AddParam("linktitle", linktitle);
				conn.ExecQuery(query);
			}
		}

		/// <summary>
		/// Remove announcement
		/// </summary>
		/// <remarks>
		/// Deletes an announcement from the system
		/// </remarks>
		/// <param name="id">Identifier of announcement to delete</param>
		public static void DeleteAnnouncement(int id)
		{
			User.RequirePrivilege(Privilege.ManageAnnouncements);
			using (DBConnection conn = DBConnector.GetConnection())
			{
				Query query = new Query(conn, @"DELETE FROM Announcements WHERE ID={id}");
				query.AddParam("id", id);
				conn.ExecQuery(query);
			}
		}

		/// <summary>
		/// Identifier
		/// </summary>
		/// <remarks>
		/// The identifier for the announcement
		/// </remarks>
		/// <value>
		/// Identifier
		/// </value>
		public int ID
		{
			get
			{
				return id;
			}
		}

		/// <summary>
		/// User ID
		/// </summary>
		/// <remarks>
		/// The user identifier of the creator of the announcement
		/// </remarks>
		/// <value>
		/// User ID
		/// </value>
		public int UserID
		{
			get
			{
				return userid;
			}
		}

		/// <summary>
		/// Posting date
		/// </summary>
		/// <remarks>
		/// The date the announcement was posted
		/// </remarks>
		/// <value>
		/// Posting date
		/// </value>
		public DateTime Posted
		{
			get
			{
				return posted;
			}
		}

		/// <summary>
		/// Title
		/// </summary>
		/// <remarks>
		/// Announcement title
		/// </remarks>
		/// <value>
		/// Title
		/// </value>
		public string Title
		{
			get
			{
				return title;
			}
		}

		/// <summary>
		/// Text
		/// </summary>
		/// <remarks>
		/// Announcement text
		/// </remarks>
		/// <value>
		/// Text
		/// </value>
		public string Text
		{
			get
			{
				return text;
			}
		}

		/// <summary>
		/// Hyperlink title
		/// </summary>
		/// <remarks>
		/// The title of the optional hyperlink
		/// </remarks>
		/// <value>
		/// Hyperlink title
		/// </value>
		public string LinkTitle
		{
			get
			{
				return linktitle;
			}
		}

		/// <summary>
		/// Hyperlink URL
		/// </summary>
		/// <remarks>
		/// The URL of the optional hyperlink
		/// </remarks>
		/// <value>
		/// Hyperlink URL
		/// </value>
		public string LinkUrl
		{
			get
			{
				return linkurl;
			}
		}
	}
}
