using System;
using System.Collections;
using System.Data;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using NUnit.Framework;

namespace Orciid.Core
{
	/// <summary>
	/// Test class for <see cref="ImportFileReader"/> class.
	/// </summary>
	/// <remarks>
	/// This class contains test methods.  It does not need to be compiled into the package.
	/// </remarks>
	[TestFixture]
	public class TestImportFileReader
	{
		/// <summary>
		/// Test
		/// </summary>
		/// <remarks>
		/// Test
		/// </remarks>
		[Test]
		public void FileTypeDetection()
		{
			Assert.AreEqual(ImportFileReader.FileType.CSV, 
				ImportFileReader.GetFileTypeByExtension("test.csv"));
			Assert.AreEqual(ImportFileReader.FileType.TSV, 
				ImportFileReader.GetFileTypeByExtension("test.tsv"));
			Assert.AreEqual(ImportFileReader.FileType.XML, 
				ImportFileReader.GetFileTypeByExtension("test.xml"));
			
			Assert.AreEqual(ImportFileReader.FileType.UNKNOWN, 
				ImportFileReader.GetFileTypeByExtension("test"));
			Assert.AreEqual(ImportFileReader.FileType.UNKNOWN, 
				ImportFileReader.GetFileTypeByExtension("test.dat"));

			Assert.AreEqual(ImportFileReader.FileType.CSV, 
				ImportFileReader.GetFileTypeByMimeType("text/csv"));
			Assert.AreEqual(ImportFileReader.FileType.CSV, 
				ImportFileReader.GetFileTypeByMimeType("text/comma-separated-values"));
			Assert.AreEqual(ImportFileReader.FileType.CSV, 
				ImportFileReader.GetFileTypeByMimeType("text/csv;charset=ISO-8859-1"));

			Assert.AreEqual(ImportFileReader.FileType.TSV, 
				ImportFileReader.GetFileTypeByMimeType("text/tsv"));
			Assert.AreEqual(ImportFileReader.FileType.TSV, 
				ImportFileReader.GetFileTypeByMimeType("text/tab-separated-values"));
			Assert.AreEqual(ImportFileReader.FileType.TSV, 
				ImportFileReader.GetFileTypeByMimeType("text/tab-separated-values;charset=ISO-8859-1"));

			Assert.AreEqual(ImportFileReader.FileType.XML, 
				ImportFileReader.GetFileTypeByMimeType("text/xml"));
			Assert.AreEqual(ImportFileReader.FileType.XML, 
				ImportFileReader.GetFileTypeByMimeType("text/xml;charset=ISO-8859-1"));

			Assert.AreEqual(ImportFileReader.FileType.UNKNOWN, 
				ImportFileReader.GetFileTypeByMimeType("text/plain"));
			Assert.AreEqual(ImportFileReader.FileType.UNKNOWN, 
				ImportFileReader.GetFileTypeByMimeType("image/gif"));
			Assert.AreEqual(ImportFileReader.FileType.UNKNOWN, 
				ImportFileReader.GetFileTypeByMimeType("image/gif;charset=ISO-8859-1"));
		}

        /// <summary>
        /// Test
        /// </summary>
        /// <remarks>
        /// Test
        /// </remarks>
		[Test]
		public void ImportCSVDataFile()
		{
			string csvfile = 
@"ID,Resource,Title,Creator,View
A1,A1.jpg,Title A,Creator A,View A1
A2,A2.jpg,Title A,Creator A,View A2
B1,B1.jpg,Title B,Creator B,View B1
";

			ImportFileReader r = new ImportFileReader(
				new StringReader(csvfile), ImportFileReader.FileType.CSV);
			DataTable table = r.GetData();
			CheckTable(table, false);
		}

        /// <summary>
        /// Test
        /// </summary>
        /// <remarks>
        /// Test
        /// </remarks>
		[Test]
		public void ImportTSVDataFile()
		{
			string tsvfile = 				
"ID\tResource\tTitle\tCreator\tView\n" +
"A1\tA1.jpg\tTitle A\tCreator A\tView A1\n" +
"A2\tA2.jpg\tTitle A\tCreator A\tView A2\n" +
"B1\tB1.jpg\tTitle B\tCreator B\tView B1\n";
			ImportFileReader r = new ImportFileReader(
				new StringReader(tsvfile), ImportFileReader.FileType.TSV);
			DataTable table = r.GetData();
			CheckTable(table, false);
		}

        /// <summary>
        /// Test
        /// </summary>
        /// <remarks>
        /// Test
        /// </remarks>
		[Test]
		public void ImportSimpleXMLDataFile()
		{
			string simplexmlfile =
@"<data>
<record Resource='A1.jpg'>
	<ID>A1</ID>
	<Title>Title A</Title>
	<Creator>Creator A</Creator>
	<View>View A1</View>
</record>
<record Resource='A2.jpg'>
	<ID>A2</ID>
	<Title>Title A</Title>
	<Creator>Creator A</Creator>
	<View>View A2</View>
</record>
<record Resource='B1.jpg'>
	<ID>B1</ID>
	<Title>Title B</Title>
	<Creator>Creator B</Creator>
	<View>View B1</View>
</record>
</data>
";
			ImportFileReader r = new ImportFileReader(
				new StringReader(simplexmlfile), ImportFileReader.FileType.XML);
			DataTable table = r.GetData();
			CheckTable(table, false);
		}

        /// <summary>
        /// Test
        /// </summary>
        /// <remarks>
        /// Test
        /// </remarks>
		[Test]
		public void ImportComplexXMLDataFile()
		{
			string complexxmlfile =
@"<data>
<work>
	<Title>Title A</Title>
	<Creator>Creator A</Creator>
	<image Resource='A1.jpg'>
		<ID>A1</ID>
		<View>View A1</View>
	</image>
	<image Resource='A2.jpg'>
		<ID>A2</ID>
		<View>View A2</View>
	</image>
</work>
<work>
	<Title>Title B</Title>
	<Creator>Creator B</Creator>
	<image Resource='B1.jpg'>
		<ID>B1</ID>
		<View>View B1</View>
	</image>
</work>
</data>
";
			ImportFileReader r = new ImportFileReader(
				new StringReader(complexxmlfile), ImportFileReader.FileType.XML);
			DataTable table = r.GetData();
			CheckTable(table, true);
		}

		private void CheckTable(DataTable table, bool imageprefix)
		{
			string ID = imageprefix ? "image.ID" : "ID";
			string Resource = imageprefix ? "image.Resource" : "Resource";
			string View = imageprefix ? "image.View" : "View";
			
			Assert.IsNotNull(table);

			PrintColumns(table);

			Assert.Greater(table.Columns.Count, 4);			
			Assert.IsTrue(table.Columns.Contains(ID));
			Assert.IsTrue(table.Columns.Contains(Resource));
			Assert.IsTrue(table.Columns.Contains("Title"));
			Assert.IsTrue(table.Columns.Contains("Creator"));
			Assert.IsTrue(table.Columns.Contains(View));
			
			Assert.AreEqual(3, table.Rows.Count);

			Assert.AreEqual("A1", table.Rows[0][ID].ToString());
			Assert.AreEqual("A1.jpg", table.Rows[0][Resource].ToString());
			Assert.AreEqual("Title A", table.Rows[0]["Title"].ToString());
			Assert.AreEqual("Creator A", table.Rows[0]["Creator"].ToString());
			Assert.AreEqual("View A1", table.Rows[0][View].ToString());

			Assert.AreEqual("A2", table.Rows[1][ID].ToString());
			Assert.AreEqual("A2.jpg", table.Rows[1][Resource].ToString());
			Assert.AreEqual("Title A", table.Rows[1]["Title"].ToString());
			Assert.AreEqual("Creator A", table.Rows[1]["Creator"].ToString());
			Assert.AreEqual("View A2", table.Rows[1][View].ToString());

			Assert.AreEqual("B1", table.Rows[2][ID].ToString());
			Assert.AreEqual("B1.jpg", table.Rows[2][Resource].ToString());
			Assert.AreEqual("Title B", table.Rows[2]["Title"].ToString());
			Assert.AreEqual("Creator B", table.Rows[2]["Creator"].ToString());
			Assert.AreEqual("View B1", table.Rows[2][View].ToString());
		}

		/// <summary>
		/// Test
		/// </summary>
		/// <remarks>
		/// Test
		/// </remarks>
		[Test]
		public void DropAttributesOnDataElements()
		{
			string simplexmlfile =
@"<data>
<record Resource='A1.jpg' ID='A1'>
	<Title Refinement='display'>Title A</Title>
	<Creator>Creator A</Creator>
	<View Name='main'>View A1</View>
</record>
</data>
";
			ImportFileReader r = new ImportFileReader(
				new StringReader(simplexmlfile), ImportFileReader.FileType.XML);
			DataTable table = r.GetData();
			Assert.IsFalse(table.Columns.Contains("Title.Refinement"));
			Assert.IsFalse(table.Columns.Contains("View.Name"));
		}

		/// <summary>
		/// Test
		/// </summary>
		/// <remarks>
		/// Test
		/// </remarks>
		[Test]
		public void VireoCatData()
		{			
			string xml = 
#region data
@"<?xml version='1.0' encoding='UTF-8'?>
<vra xmlns='http://www.vraweb.org/vracore4.htm'>
<work xmlns='' id='W6' source='VireoCat Test Database, Susan Jane Williams (consultant)' refid='6'>
<agentSet>
<display>unknown, Greek (ancient )</display>
<agent>
<name type='Personal'>unknown</name>
<dates type=''>
<earliestDate/>
<latestDate/>
</dates>
<culture/>
<role vocab='' refid=''/>
<attribution/>
</agent>
</agentSet>
<dateSet>
<display>ca. 490-480 BCE</display>
<date type=''>
<earliestDate>-0490</earliestDate>
<latestDate>-0480</latestDate>
</date>
</dateSet>
<descriptionSet>
<description>Discovered 1811.</description>
</descriptionSet>
<locationSet>
<display>Glyptothek, Munich, Bavaria, Germany ()
Temple of Aphaia, Aegina, Central Greece and Euboea, Greece ()</display>
<location type='formerGeographic'>
<name type='geographic' vocab='DOA' refid='' extent='building'>Temple of Aphaia</name>
<name type='geographic' vocab='TGN' refid='Aegina' extent='inhabited place'>Aegina</name>
<name type='geographic' vocab='TGN' refid='' extent='region'>Central Greece and Euboea</name>
<name type='geographic' vocab='' refid='' extent='nation'>Greece</name>
<name type='geographic' vocab='' refid='' extent='continent'>Europe</name>
</location>
<location type='repository'>
<name type='geographic' vocab='DOA' refid='' extent='building'>Glyptothek</name>
<name type='geographic' vocab='TGN' refid='Munich' extent='inhabited place'>Munich</name>
<name type='geographic' vocab='TGN' refid='' extent='region'>Bavaria</name>
<name type='geographic' vocab='' refid='' extent='nation'>Germany</name>
<name type='geographic' vocab='' refid='' extent='continent'>Europe</name>
</location>
<location type=''/>
</locationSet>
<materialSet>
<display>parian marble</display>
</materialSet>
<measurementsSet>
<display/>
</measurementsSet>
<relationSet>
<display/>
<relation type='part of' refid='6' relids='22' source=''>East Pediment, Temple of Aphaia</relation>
<relation type='larger entity' refid='6' relids='20' source=''>Temple of Aphaia</relation>
</relationSet>
<rightsSet>
<display/>
<rights/>
</rightsSet>
<sourceSet>
<display/>
<source>
<name type='Vendor'/>
<refid type=''/>
</source>
<source>
<name type='Vendor'/>
<refid type=''/>
</source>
<source>
<name type='Vendor'/>
<refid type=''/>
</source>
<source>
<name type='Vendor'/>
<refid type=''/>
</source>
</sourceSet>
<stateEditionSet>
<display/>
<stateEdition type='' source=''>
<name/>
</stateEdition>
</stateEditionSet>
<stylePeriodSet>
<display>Late Archaic </display>
<stylePeriod vocab='AAT' refid='300106995'>Late Archaic </stylePeriod>
<stylePeriod vocab='' refid=''/>
</stylePeriodSet>
<subjectSet>
<subject>
<term type='' source='' refid=''>War</term>
</subject>
</subjectSet>
<techniqueSet>
<display>carving (processes) </display>
<technique vocab='AAT' refid='300053149'>carving (processes) </technique>
</techniqueSet>
<textrefSet>
<display/>
<textref>
<name type=''/>
</textref>
<notes/>
</textrefSet>
<titleSet>
<display>Dying Warrior from East Pediment; local; English; alternate
Fallen Warrior; local; English; preferred</display>
<title type='local' pref='alternate' lang='English'>Dying Warrior from East Pediment</title>
<title type='local' pref='preferred' lang='English'>Fallen Warrior</title>
</titleSet>
<worktypeSet>
<display>sculpture</display>
<worktype>sculpture</worktype>
</worktypeSet>
<inscriptionSet>
<display/>
</inscriptionSet>
<image refid='10005' source='Your Institution Name'>
<titleSet>
<display>Overall view of warrior with shield</display>
<title type='descriptive'>Overall view of warrior with shield</title>
</titleSet>
<relationSet>
<relation type='image-of' refid='6' source='VireoCat Test Database, Susan Jane Williams (consultant)'/>
</relationSet>
<worktypeSet>
<display>Sculpture</display>
</worktypeSet>
</image>
<image refid='10006' source='Your Institution Name'>
<titleSet>
<display>Detail view of warrior with shield</display>
<title type='descriptive'>Detail view of warrior with shield</title>
</titleSet>
<relationSet>
<relation type='image-of' refid='6' source='VireoCat Test Database, Susan Jane Williams (consultant)'/>
</relationSet>
<worktypeSet>
<display>Sculpture</display>
</worktypeSet>
</image>
<image refid='10007' source='Your Institution Name'>
<titleSet>
<display>Detail view, head of warrior with shield</display>
<title type='descriptive'>Detail view, head of warrior with shield</title>
</titleSet>
<relationSet>
<relation type='image-of' refid='6' source='VireoCat Test Database, Susan Jane Williams (consultant)'/>
</relationSet>
<worktypeSet>
<display>Sculpture</display>
</worktypeSet>
</image>
<image refid='10028' source='Your Institution Name'>
<titleSet>
<display>Overall view of warrior with shield</display>
<title type='descriptive'>Overall view of warrior with shield</title>
</titleSet>
<relationSet>
<relation type='image-of' refid='6' source='VireoCat Test Database, Susan Jane Williams (consultant)'/>
</relationSet>
<worktypeSet>
<display>Sculpture</display>
</worktypeSet>
</image>
</work>
<work xmlns='' id='W7' source='VireoCat Test Database, Susan Jane Williams (consultant)' refid='7'>
<agentSet>
<display>Gustav Klucis (Latvian designer, 1895-1944)</display>
<agent>
<name type='Personal'>Klucis, Gustav</name>
<dates type=''>
<earliestDate>1895</earliestDate>
<latestDate>1944</latestDate>
</dates>
<culture>Latvian</culture>
<role vocab='' refid=''>designer</role>
<attribution/>
</agent>
</agentSet>
<dateSet>
<display>1930</display>
<date type=''>
<earliestDate>1930</earliestDate>
<latestDate>1930</latestDate>
</date>
</dateSet>
<descriptionSet>
<description>A political poster produced in support of the first Soviet 'Five Year Plan.'</description>
</descriptionSet>
<locationSet>
<display>Museum of Modern Art, New York, New York, United States ()</display>
<location type='repository'>
<name type='geographic' vocab='DOA' refid='' extent='building'>Museum of Modern Art</name>
<name type='geographic' vocab='TGN' refid='New York' extent='inhabited place'>New York</name>
<name type='geographic' vocab='TGN' refid='' extent='state'>New York</name>
<name type='geographic' vocab='' refid='' extent='nation'>United States</name>
<name type='geographic' vocab='' refid='' extent='continent'>North and Central America</name>
</location>
</locationSet>
<materialSet>
<display>printers ink on paper</display>
</materialSet>
<measurementsSet>
<display>102.5 x 72 cm</display>
<measurements type='height' unit='cm' extent=''>102.5</measurements>
<measurements type='length' unit='cm' extent=''>72</measurements>
</measurementsSet>
<relationSet>
<display/>
<relation type='' refid='7' relids='' source=''/>
</relationSet>
<rightsSet>
<display/>
<rights/>
</rightsSet>
<sourceSet>
<display/>
<source>
<name type='Book'>Russian Constructivism</name>
<refid type='ISBN'>0300034067</refid>
</source>
</sourceSet>
<stateEditionSet>
<display/>
<stateEdition type='' source=''>
<name/>
</stateEdition>
</stateEditionSet>
<stylePeriodSet>
<display>Constructivist</display>
<stylePeriod vocab='AAT' refid='300021393'>Constructivist</stylePeriod>
<stylePeriod vocab='' refid=''/>
</stylePeriodSet>
<subjectSet>
<subject>
<term type='' source='' refid=''>Mining</term>
<term type='conceptTopic' source='' refid=''>Communism</term>
<term type='' source='' refid=''>Soviet Union--History</term>
<term type='' source='' refid=''>Workers in art</term>
</subject>
</subjectSet>
<techniqueSet>
<display>printing, offset</display>
<technique vocab='AAT' refid=''>printing, offset</technique>
</techniqueSet>
<textrefSet>
<display/>
<textref>
<name type=''/>
</textref>
<notes/>
</textrefSet>
<titleSet>
<display>Vernem ugol' nyi dolg strane; creator; Russian; preferred
We shall repay the Coal Debt to our Country; translated; English; alternate</display>
<title type='translated' pref='alternate' lang='English'>We shall repay the Coal Debt to our Country</title>
<title type='creator' pref='preferred' lang='Russian'>Vernem ugol' nyi dolg strane</title>
</titleSet>
<worktypeSet>
<display>postersprints; planographic prints; lithographs</display>
<worktype>posters</worktype>
<worktype>lithographs</worktype>
<worktype/>
</worktypeSet>
<inscriptionSet>
<display/>
</inscriptionSet>
<image refid='10031' source='Your Institution Name'>
<titleSet>
<display>Overall view</display>
<title type='descriptive'>Overall view</title>
</titleSet>
<relationSet>
<relation type='image-of' refid='7' source='VireoCat Test Database, Susan Jane Williams (consultant)'/>
</relationSet>
<worktypeSet>
<display>Commercial Arts</display>
</worktypeSet>
</image>
</work>
<work xmlns='' id='W8' source='VireoCat Test Database, Susan Jane Williams (consultant)' refid='8'>
<agentSet>
<display>Alexandra Exter (Russian costume designer, 1884-1949) and
Yakov Alexandrovitch Protazanov (Russian director, 1881-1945)</display>
<agent>
<name type='Personal'>Protazanov, Yakov Alexandrovitch</name>
<dates type=''>
<earliestDate>1881</earliestDate>
<latestDate>1945</latestDate>
</dates>
<culture>Russian</culture>
<role vocab='' refid=''>director</role>
<attribution/>
</agent>
<agent>
<name type='Personal'>Exter, Alexandra</name>
<dates type=''>
<earliestDate>1884</earliestDate>
<latestDate>1949</latestDate>
</dates>
<culture>Russian</culture>
<role vocab='' refid=''>costume designer</role>
<attribution/>
</agent>
</agentSet>
<dateSet>
<display>1924</display>
<date type=''>
<earliestDate>1924</earliestDate>
<latestDate>1924</latestDate>
</date>
</dateSet>
<descriptionSet>
<description>Exter  [Ekster] created the costume designs;  I. Rabinovich created the models used for the Martian city; sets were by V. Simov. The film was based on the novel by Aleksei Tolstoi and produced and directed by Yakov Protazanov.</description>
</descriptionSet>
<locationSet>
<display>British Film Institute, London, England, United Kingdom ()</display>
<location type='repository'>
<name type='geographic' vocab='DOA' refid='' extent='organization'>British Film Institute</name>
<name type='geographic' vocab='TGN' refid='London' extent='inhabited place'>London</name>
<name type='geographic' vocab='TGN' refid='' extent='country'>England</name>
<name type='geographic' vocab='' refid='' extent='nation'>United Kingdom</name>
<name type='geographic' vocab='' refid='' extent='continent'>Europe</name>
</location>
</locationSet>
<materialSet>
<display>motion picture film</display>
</materialSet>
<measurementsSet>
<display>111 min. running time</display>
<measurements type='running time' unit='min' extent=''>111</measurements>
</measurementsSet>
<relationSet>
<display/>
</relationSet>
<rightsSet>
<display/>
<rights/>
</rightsSet>
<sourceSet>
<display>Lodder, Christina; Russian Constructivism, Yale University Press, New Haven CT, 1983 </display>
<source>
<name type='Book'>Russian Constructivism</name>
<refid type='ISBN'>0300034067</refid>
</source>
</sourceSet>
<stateEditionSet>
<display/>
<stateEdition type='' source=''>
<name/>
</stateEdition>
</stateEditionSet>
<stylePeriodSet>
<display>Constructivist</display>
<stylePeriod vocab='AAT' refid='300021393'>Constructivist</stylePeriod>
<stylePeriod vocab='' refid=''/>
</stylePeriodSet>
<subjectSet>
<subject>
<term type='conceptTopic' source='' refid=''>Future life</term>
<term type='' source='' refid=''>Martians</term>
</subject>
</subjectSet>
<techniqueSet>
<display>filmmaking</display>
<technique vocab='AAT' refid=''>filmmaking</technique>
</techniqueSet>
<textrefSet>
<display/>
<textref>
<name type=''/>
</textref>
<notes/>
</textrefSet>
<titleSet>
<display>Aelita [motion picture]; local; English; preferred
Aelita, the Queen of Mars; local; English; alternate</display>
<title type='local' pref='alternate' lang='English'>Aelita, the Queen of Mars</title>
<title type='local' pref='preferred' lang='English'>Aelita [motion picture]</title>
</titleSet>
<worktypeSet>
<display>motion pictures (visual works); feature films</display>
<worktype>feature films</worktype>
<worktype/>
</worktypeSet>
<inscriptionSet>
<display/>
</inscriptionSet>
<image refid='10029' source='Your Institution Name'>
<titleSet>
<display>Film still showing characters Ikushka and Martian officer</display>
<title type='descriptive'>Film still showing characters Ikushka and Martian officer</title>
</titleSet>
<relationSet>
<relation type='image-of' refid='8' source='VireoCat Test Database, Susan Jane Williams (consultant)'/>
</relationSet>
<worktypeSet>
<display>Film and Video</display>
</worktypeSet>
</image>
</work>
<work xmlns='' id='W9' source='VireoCat Test Database, Susan Jane Williams (consultant)' refid='9'>
<agentSet>
<display>Il'ya Chashnik (Russian designer, 1902-1929);
Kazimir Malevich  (Russian designer, 1878-1935);
Nikolay Suetin (Russian designer, 1897-1954)</display>
<agent>
<name type='Personal'>Malevich, Kazimir</name>
<dates type=''>
<earliestDate>1878</earliestDate>
<latestDate>1935</latestDate>
</dates>
<culture>Russian</culture>
<role vocab='' refid=''>designer</role>
<attribution/>
</agent>
<agent>
<name type='Personal'>Suetin, Nikolay </name>
<dates type=''>
<earliestDate>1897</earliestDate>
<latestDate>1954</latestDate>
</dates>
<culture>Russian</culture>
<role vocab='' refid=''>designer</role>
<attribution/>
</agent>
<agent>
<name type='Personal'>Chashnik, Il'ya</name>
<dates type=''>
<earliestDate>1902</earliestDate>
<latestDate>1929</latestDate>
</dates>
<culture>Russian</culture>
<role vocab='' refid=''>designer</role>
<attribution/>
</agent>
</agentSet>
<dateSet>
<display>ca. 1922</display>
<date type='creation'>
<earliestDate>1920</earliestDate>
<latestDate>1925</latestDate>
</date>
</dateSet>
<descriptionSet>
<description>Various pieces of porcelain as shown reproduced in the periodical Russkoe iskusstvo, No. 2/3, 1923.</description>
</descriptionSet>
<locationSet>
<display>Moscow, Rossiya, Russia ()</display>
<location type='creationGeographic'>
<name type='geographic' vocab='TGN' refid='Moscow' extent='inhabited place'>Moscow</name>
<name type='geographic' vocab='TGN' refid='' extent='region'>Rossiya</name>
<name type='geographic' vocab='' refid='' extent='nation'>Russia</name>
<name type='geographic' vocab='' refid='' extent='continent'>Asia</name>
</location>
</locationSet>
<materialSet>
<display>porcelain</display>
</materialSet>
<measurementsSet>
<display/>
</measurementsSet>
<relationSet>
<display/>
</relationSet>
<rightsSet>
<display/>
<rights/>
</rightsSet>
<sourceSet>
<display>Lodder, Christina; Russian Constructivism, Yale University Press, New Haven CT, 1983 </display>
<source>
<name type='Book'>Russian Constructivism</name>
<refid type='ISBN'>0300034067</refid>
</source>
</sourceSet>
<stateEditionSet>
<display/>
<stateEdition type='' source=''>
<name/>
</stateEdition>
</stateEditionSet>
<stylePeriodSet>
<display>Constructivist</display>
<stylePeriod vocab='AAT' refid='300021393'>Constructivist</stylePeriod>
<stylePeriod vocab='' refid=''/>
</stylePeriodSet>
<subjectSet>
<subject>
<term type='' source='' refid=''>Tea</term>
<term type='' source='' refid=''>Material Culture</term>
</subject>
</subjectSet>
<techniqueSet>
<display>fabrication attributes: ceramics; molded</display>
<technique vocab='AAT' refid=''>molded</technique>
<technique vocab='AAT' refid='300263319'>fabrication attributes: ceramics</technique>
</techniqueSet>
<textrefSet>
<display/>
<textref>
<name type=''/>
</textref>
<notes/>
</textrefSet>
<titleSet>
<display>Suprematist Porcelain Designs; translated; English; preferred</display>
<title type='translated' pref='preferred' lang='English'>Suprematist Porcelain Designs</title>
</titleSet>
<worktypeSet>
<display>containers; vessels; plates (dishes)containers; drinking vessels; cupscontainers; coffeepotstableware; dinnerware</display>
<worktype>dinnerware</worktype>
<worktype>plates (dishes)</worktype>
<worktype>coffeepots</worktype>
<worktype>cups</worktype>
</worktypeSet>
<inscriptionSet>
<display/>
</inscriptionSet>
<image refid='10030' source='Your Institution Name'>
<titleSet>
<display>Various pieces of porcelain [shown in Russkoe iskusstvo, No. 2/3, 1923]</display>
<title type='descriptive'>Various pieces of porcelain [shown in Russkoe iskusstvo, No. 2/3, 1923]</title>
</titleSet>
<relationSet>
<relation type='image-of' refid='9' source='VireoCat Test Database, Susan Jane Williams (consultant)'/>
</relationSet>
<worktypeSet>
<display>Applied Arts: Ceramics</display>
</worktypeSet>
</image>
</work>
<work xmlns='' id='W10' source='VireoCat Test Database, Susan Jane Williams (consultant)' refid='10'>
<agentSet>
<display>unknown, American</display>
<agent>
<name type='Personal'>unknown</name>
<dates type=''>
<earliestDate/>
<latestDate/>
</dates>
<culture/>
<role vocab='' refid=''/>
<attribution/>
</agent>
</agentSet>
<dateSet>
<display>ca. 1970</display>
<date type=''>
<earliestDate>1965</earliestDate>
<latestDate>1975</latestDate>
</date>
</dateSet>
<descriptionSet>
<description/>
</descriptionSet>
<locationSet>
<display/>
</locationSet>
<materialSet>
<display/>
</materialSet>
<measurementsSet>
<display/>
</measurementsSet>
<relationSet>
<display/>
</relationSet>
<rightsSet>
<display/>
<rights/>
</rightsSet>
<sourceSet>
<display/>
<source>
<name type='Article (Serial)'>Echoes</name>
<refid type='ISSN'>1089-7046</refid>
</source>
</sourceSet>
<stateEditionSet>
<display/>
<stateEdition type='' source=''>
<name/>
</stateEdition>
</stateEditionSet>
<stylePeriodSet>
<display/>
<stylePeriod vocab='' refid=''/>
</stylePeriodSet>
<subjectSet>
<subject>
<term type='' source='' refid=''>Breuer, Marcel, 1902-1981</term>
</subject>
</subjectSet>
<techniqueSet>
<display>photography, black and white</display>
<technique vocab='AAT' refid=''>photography, black and white</technique>
</techniqueSet>
<textrefSet>
<display/>
<textref>
<name type=''/>
</textref>
<notes/>
</textrefSet>
<titleSet>
<display>Photograph of Marcel Breuer; local; English; preferred</display>
<title type='local' pref='preferred' lang='English'>Photograph of Marcel Breuer</title>
</titleSet>
<worktypeSet>
<display>photographs; positives; photographic prints</display>
<worktype>photographic prints</worktype>
</worktypeSet>
<inscriptionSet>
<display/>
</inscriptionSet>
<image refid='10013' source='Your Institution Name'>
<titleSet>
<display>Something</display>
<title type='descriptive'>Something</title>
</titleSet>
<relationSet>
<relation type='image-of' refid='10' source='VireoCat Test Database, Susan Jane Williams (consultant)'/>
</relationSet>
<worktypeSet>
<display>collateral material</display>
</worktypeSet>
</image>
</work>
</vra>
";
			#endregion
			ImportFileReader r = new ImportFileReader(
				new StringReader(xml), ImportFileReader.FileType.XML);
			DataTable table = r.GetData();
			PrintColumns(table);
		}

		private void PrintColumns(DataTable table)
		{
			Console.WriteLine("Columns:");
			SortedList list = new SortedList();
			foreach (DataColumn col in table.Columns)
				list.Add(col.Caption, null);
			foreach (string c in list.Keys)
				Console.WriteLine("- " + c);
		}

		/// <summary>
		/// Test
		/// </summary>
		/// <remarks>
		/// Test
		/// </remarks>
		[Test]
		public void RepeatedXmlElements()
		{
			string xml =
@"<data>
<record id='One'>
	<title>First</title>
	<title>Second</title>
	<creator>Alpha</creator>
	<creator>Beta</creator>
</record>
<record id='Two'>
	<title>Third</title>
	<title>Fourth</title>
	<creator>Gamma</creator>
	<creator>Delta</creator>
</record>
</data>
";	
			ImportFileReader r = new ImportFileReader(
				new StringReader(xml), ImportFileReader.FileType.XML);
			DataTable table = r.GetData();
			PrintColumns(table);

			Assert.AreEqual(3, table.Columns.Count);
			Assert.IsTrue(table.Columns.Contains("id"));
			Assert.IsTrue(table.Columns.Contains("title"));
			Assert.IsTrue(table.Columns.Contains("creator"));
			
			Assert.AreEqual(4, table.Rows.Count);

			Assert.AreEqual("One", table.Rows[0]["id"]);
			Assert.AreEqual(System.DBNull.Value, table.Rows[1]["id"]);
			Assert.AreEqual("Two", table.Rows[2]["id"]);
			Assert.AreEqual(System.DBNull.Value, table.Rows[3]["id"]);

			Assert.AreEqual("First", table.Rows[0]["title"]);
			Assert.AreEqual("Second", table.Rows[1]["title"]);
			Assert.AreEqual("Third", table.Rows[2]["title"]);
			Assert.AreEqual("Fourth", table.Rows[3]["title"]);

			Assert.AreEqual("Alpha", table.Rows[0]["creator"]);
			Assert.AreEqual("Beta", table.Rows[1]["creator"]);
			Assert.AreEqual("Gamma", table.Rows[2]["creator"]);
			Assert.AreEqual("Delta", table.Rows[3]["creator"]);
		}

		/// <summary>
		/// Test
		/// </summary>
		/// <remarks>
		/// Test
		/// </remarks>
		[Test]
		public void ImportAndSplitValuesStatic()
		{
			string csvfile = 
@"ID,Resource,Title,Creator,View
A1,A1.jpg,Title A1*Title A2,Creator A,View A1
A2,A2.jpg,Title A1*Title A2,Creator A1*Creator A2,View A2
B1,B1.jpg,Title B,Creator B,View B1
";

			ImportFileReader r = new ImportFileReader(
				new StringReader(csvfile), ImportFileReader.FileType.CSV);
			r.SplitMode = ImportFileReader.ValueSplitMode.Static;
			r.SplitAt = "*";
			DataTable table = r.GetData();
			
			Assert.IsNotNull(table);

			PrintColumns(table);

			Assert.AreEqual(table.Columns.Count, 5);			
			Assert.IsTrue(table.Columns.Contains("ID"));
			Assert.IsTrue(table.Columns.Contains("Resource"));
			Assert.IsTrue(table.Columns.Contains("Title"));
			Assert.IsTrue(table.Columns.Contains("Creator"));
			Assert.IsTrue(table.Columns.Contains("View"));
			
			Assert.AreEqual(5, table.Rows.Count);

			Assert.AreEqual("A1", table.Rows[0]["ID"].ToString());
			Assert.AreEqual("A1.jpg", table.Rows[0]["Resource"].ToString());
			Assert.AreEqual("Title A1", table.Rows[0]["Title"].ToString());
			Assert.AreEqual("Creator A", table.Rows[0]["Creator"].ToString());
			Assert.AreEqual("View A1", table.Rows[0]["View"].ToString());

			Assert.AreEqual("", table.Rows[1]["ID"].ToString());
			Assert.AreEqual("", table.Rows[1]["Resource"].ToString());
			Assert.AreEqual("Title A2", table.Rows[1]["Title"].ToString());
			Assert.AreEqual("", table.Rows[1]["Creator"].ToString());
			Assert.AreEqual("", table.Rows[1]["View"].ToString());
			
			Assert.AreEqual("A2", table.Rows[2]["ID"].ToString());
			Assert.AreEqual("A2.jpg", table.Rows[2]["Resource"].ToString());
			Assert.AreEqual("Title A1", table.Rows[2]["Title"].ToString());
			Assert.AreEqual("Creator A1", table.Rows[2]["Creator"].ToString());
			Assert.AreEqual("View A2", table.Rows[2]["View"].ToString());

			Assert.AreEqual("", table.Rows[3]["ID"].ToString());
			Assert.AreEqual("", table.Rows[3]["Resource"].ToString());
			Assert.AreEqual("Title A2", table.Rows[3]["Title"].ToString());
			Assert.AreEqual("Creator A2", table.Rows[3]["Creator"].ToString());
			Assert.AreEqual("", table.Rows[3]["View"].ToString());

			Assert.AreEqual("B1", table.Rows[4]["ID"].ToString());
			Assert.AreEqual("B1.jpg", table.Rows[4]["Resource"].ToString());
			Assert.AreEqual("Title B", table.Rows[4]["Title"].ToString());
			Assert.AreEqual("Creator B", table.Rows[4]["Creator"].ToString());
			Assert.AreEqual("View B1", table.Rows[4]["View"].ToString());
		}

		/// <summary>
		/// Test
		/// </summary>
		/// <remarks>
		/// Test
		/// </remarks>
		[Test]
		public void ImportAndSplitValuesRegex()
		{
			string csvfile = 
				@"ID,Resource,Title,Creator,View
A1,A1.jpg,Title A1&Title A2,Creator A,View A1
A2,A2.jpg,Title A1$Title A2,Creator A1*split here*Creator A2,View A2
B1,B1.jpg,Title B,Creator B,View B1
";

			ImportFileReader r = new ImportFileReader(
				new StringReader(csvfile), ImportFileReader.FileType.CSV);
			r.SplitMode = ImportFileReader.ValueSplitMode.Regex;
			r.SplitAt = @"[&$]|(\*split here\*)";
			DataTable table = r.GetData();
			
			Assert.IsNotNull(table);

			PrintColumns(table);

			Assert.AreEqual(table.Columns.Count, 5);			
			Assert.IsTrue(table.Columns.Contains("ID"));
			Assert.IsTrue(table.Columns.Contains("Resource"));
			Assert.IsTrue(table.Columns.Contains("Title"));
			Assert.IsTrue(table.Columns.Contains("Creator"));
			Assert.IsTrue(table.Columns.Contains("View"));
			
			Assert.AreEqual(5, table.Rows.Count);

			Assert.AreEqual("A1", table.Rows[0]["ID"].ToString());
			Assert.AreEqual("A1.jpg", table.Rows[0]["Resource"].ToString());
			Assert.AreEqual("Title A1", table.Rows[0]["Title"].ToString());
			Assert.AreEqual("Creator A", table.Rows[0]["Creator"].ToString());
			Assert.AreEqual("View A1", table.Rows[0]["View"].ToString());

			Assert.AreEqual("", table.Rows[1]["ID"].ToString());
			Assert.AreEqual("", table.Rows[1]["Resource"].ToString());
			Assert.AreEqual("Title A2", table.Rows[1]["Title"].ToString());
			Assert.AreEqual("", table.Rows[1]["Creator"].ToString());
			Assert.AreEqual("", table.Rows[1]["View"].ToString());
			
			Assert.AreEqual("A2", table.Rows[2]["ID"].ToString());
			Assert.AreEqual("A2.jpg", table.Rows[2]["Resource"].ToString());
			Assert.AreEqual("Title A1", table.Rows[2]["Title"].ToString());
			Assert.AreEqual("Creator A1", table.Rows[2]["Creator"].ToString());
			Assert.AreEqual("View A2", table.Rows[2]["View"].ToString());

			Assert.AreEqual("", table.Rows[3]["ID"].ToString());
			Assert.AreEqual("", table.Rows[3]["Resource"].ToString());
			Assert.AreEqual("Title A2", table.Rows[3]["Title"].ToString());
			Assert.AreEqual("Creator A2", table.Rows[3]["Creator"].ToString());
			Assert.AreEqual("", table.Rows[3]["View"].ToString());

			Assert.AreEqual("B1", table.Rows[4]["ID"].ToString());
			Assert.AreEqual("B1.jpg", table.Rows[4]["Resource"].ToString());
			Assert.AreEqual("Title B", table.Rows[4]["Title"].ToString());
			Assert.AreEqual("Creator B", table.Rows[4]["Creator"].ToString());
			Assert.AreEqual("View B1", table.Rows[4]["View"].ToString());
		}

		/// <summary>
		/// Test
		/// </summary>
		/// <remarks>
		/// Test
		/// </remarks>
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void InvalidSplitRegex()
		{
			ImportFileReader r = new ImportFileReader(
				new StringReader("header\nrow1"), ImportFileReader.FileType.CSV);
			r.SplitMode = ImportFileReader.ValueSplitMode.Regex;
			r.SplitAt = "(";
			DataTable table = r.GetData();
		}

		private void TestEncodingFileImport(System.Text.Encoding encoding)
		{
			string file = Path.GetTempFileName();
			try
			{
				using (StreamWriter writer = new StreamWriter(file, false, encoding))
				{
					writer.WriteLine("ID,Creator");
					writer.WriteLine("A1,\"Torres-Garca, Joaqun\"");
				}

				ImportFileReader r = new ImportFileReader(file, ImportFileReader.FileType.CSV);
				DataTable table = r.GetData();
				Assert.AreEqual("A1", table.Rows[0]["ID"].ToString());
				Assert.AreEqual("Torres-Garca, Joaqun", table.Rows[0]["Creator"].ToString());
			}
			finally
			{
				if (File.Exists(file))
					File.Delete(file);
			}
		}
		

		/// <summary>
		/// Test
		/// </summary>
		/// <remarks>
		/// Test
		/// </remarks>
		[Test]
		public void ANSIFileImport()
		{
			TestEncodingFileImport(System.Text.Encoding.Default);
		}

		/// <summary>
		/// Test
		/// </summary>
		/// <remarks>
		/// Test
		/// </remarks>
		[Test]
		public void UTF8FileImport()
		{
			TestEncodingFileImport(System.Text.Encoding.UTF8);
		}

		/// <summary>
		/// Test
		/// </summary>
		/// <remarks>
		/// Test
		/// </remarks>
		[Test]
		public void UnicodeFileImport()
		{
			TestEncodingFileImport(System.Text.Encoding.Unicode);
		}
	}
}
