using NUnit.Framework;
namespace DotNetMock.Tests
{
	[TestFixture]
	public class ExpectationStringQueueTests
	{
		private ExpectationStringQueue _expectationStringQueue = null;
		[SetUp]
		public void SetUp( )
		{
			_expectationStringQueue = new ExpectationStringQueue( "ExpectationQueueString" );
		}
		[TearDown]
		public void TearDown( )
		{
			_expectationStringQueue = null;
		}
		[Test]
			public void NoExpectations()
		{
			_expectationStringQueue.Verify();
		}
		[Test]
		public void SetActual( )
		{
			string test1 = "test string 1";
			string test2 = "test string 2";
			string actualString;
			Assert.IsNull( _expectationStringQueue.Actual );
			_expectationStringQueue.Actual = test1;
			_expectationStringQueue.Actual = test2;
			actualString = _expectationStringQueue.Actual;
			Assert.IsNotNull( actualString );
			Assert.AreEqual( test1, actualString );
			actualString = _expectationStringQueue.Actual;
			Assert.IsNotNull( actualString );
			Assert.AreEqual( test2, actualString );
			Assert.IsNull( _expectationStringQueue.Actual );
		}
		[Test]
		public void SetExpected( )
		{
			string test3 = "test string 3";
			string test4 = "test string 4";
			string expectedString;
			Assert.IsNull( _expectationStringQueue.Expected );
			_expectationStringQueue.Expected = test3;
			_expectationStringQueue.Expected = test4;
			expectedString = _expectationStringQueue.Expected;
			Assert.IsNotNull( expectedString );
			Assert.AreEqual( test3, expectedString );
			expectedString = _expectationStringQueue.Expected;
			Assert.IsNotNull( expectedString );
			Assert.AreEqual( test4, expectedString );
			Assert.IsNull( _expectationStringQueue.Expected );
		}
		[Test]
		public void HasExpectations( )
		{
			string test5 = "test string 5";
			Assert.IsNull( _expectationStringQueue.Expected );
			Assert.IsFalse( _expectationStringQueue.HasExpectations );
			_expectationStringQueue.Expected = test5;
			Assert.IsTrue( _expectationStringQueue.HasExpectations );
			string expectedString = _expectationStringQueue.Expected;
			Assert.IsNotNull( expectedString );
			Assert.AreEqual( test5, expectedString );
			Assert.IsFalse( _expectationStringQueue.HasExpectations );
		}
		[Test]
		public void ClearExpected( )
		{
			string test6 = "test string 6";
			string test7 = "test string 7";
			Assert.IsNull( _expectationStringQueue.Expected );
			_expectationStringQueue.Expected = test6;
			_expectationStringQueue.Expected = test7;
			_expectationStringQueue.ClearExpected( );
			Assert.IsNull( _expectationStringQueue.Expected );
		}
		[Test]
		public void ClearActual( )
		{
			string test8 = "test string 8";
			string test9 = "test string 9";
			Assert.IsNull( _expectationStringQueue.Actual );
			_expectationStringQueue.Actual = test8;
			_expectationStringQueue.Actual = test9;
			_expectationStringQueue.ClearActual( );
			Assert.IsNull( _expectationStringQueue.Actual );
		}
		[Test]
		public void Verify( )
		{
			string test10 = "test string 10";
			string test11 = "test string 11";
			string test12 = "test string 12";
			bool exceptionFlag;

			#region Strings in queues are not equal

			_expectationStringQueue.Actual = test10;
			_expectationStringQueue.Actual = test11;
			_expectationStringQueue.Actual = test12;
			_expectationStringQueue.Expected = test10;
			_expectationStringQueue.Expected = test11;
			_expectationStringQueue.Expected = test11;
			exceptionFlag = false;
			try
			{
				_expectationStringQueue.Verify( );
			}
			catch
			{
				exceptionFlag = true;
			}
			Assert.IsTrue( exceptionFlag, "Verify() MUST fail: strings in queues are not equal" );

			#endregion

			#region Actual queue is longer than Expected one

			_expectationStringQueue.ClearActual( );
			_expectationStringQueue.ClearExpected( );
			_expectationStringQueue.Actual = test10;
			_expectationStringQueue.Actual = test11;
			_expectationStringQueue.Actual = test12;
			_expectationStringQueue.Actual = test12;
			_expectationStringQueue.Expected = test10;
			_expectationStringQueue.Expected = test11;
			_expectationStringQueue.Expected = test12;
			exceptionFlag = false;
			try
			{
				_expectationStringQueue.Verify( );
			}
			catch
			{
				exceptionFlag = true;
			}
			Assert.IsTrue( exceptionFlag, "Verify() MUST fail: Actual queue is longer than Expected one" );

			#endregion

			#region Expected queue is longer than Actual one

			_expectationStringQueue.ClearActual( );
			_expectationStringQueue.ClearExpected( );
			_expectationStringQueue.Actual = test10;
			_expectationStringQueue.Actual = test11;
			_expectationStringQueue.Actual = test12;
			_expectationStringQueue.Expected = test10;
			_expectationStringQueue.Expected = test11;
			_expectationStringQueue.Expected = test12;
			_expectationStringQueue.Expected = test12;
			exceptionFlag = false;
			try
			{
				_expectationStringQueue.Verify( );
			}
			catch
			{
				exceptionFlag = true;
			}
			Assert.IsTrue( exceptionFlag, "Verify() MUST fail: Expected queue is longer than Actual one" );

			#endregion

			#region Queues are equal

			_expectationStringQueue.ClearActual( );
			_expectationStringQueue.ClearExpected( );
			_expectationStringQueue.ClearActual( );
			_expectationStringQueue.ClearExpected( );
			_expectationStringQueue.Actual = test10;
			_expectationStringQueue.Actual = test11;
			_expectationStringQueue.Actual = test12;
			_expectationStringQueue.Expected = test10;
			_expectationStringQueue.Expected = test11;
			_expectationStringQueue.Expected = test12;
			_expectationStringQueue.Verify( );

			#endregion
		}
		[Test]
		public void FailureMode( )
		{
			string test13 = "test string 13";
			_expectationStringQueue.Expected = test13;
			Assert.IsFalse( _expectationStringQueue.ShouldCheckImmediate );
			_expectationStringQueue.VerifyImmediate = true;
			Assert.IsTrue( _expectationStringQueue.ShouldCheckImmediate );
		}
		[Test]
		public void ImmediateValidation( )
		{
			string test14 = "test string 14";
			string test15 = "test string 15";
			bool exceptionFlag;

			#region Fail case

			_expectationStringQueue.Expected = test14;
			_expectationStringQueue.Expected = test15;
			_expectationStringQueue.VerifyImmediate = true;
			exceptionFlag = false;
			try
			{
				_expectationStringQueue.Actual = test15;
			}
			catch
			{
				exceptionFlag = true;
			}
			Assert.IsTrue( exceptionFlag, "Immediate Verification MUST fail" );

			#endregion

			#region Success case

			_expectationStringQueue.ClearActual( );
			_expectationStringQueue.ClearExpected( );
			_expectationStringQueue.Expected = test14;
			_expectationStringQueue.Expected = test15;
			_expectationStringQueue.VerifyImmediate = true;
			_expectationStringQueue.Actual = test14;
			_expectationStringQueue.Actual = test15;
			_expectationStringQueue.Actual = test15;

			#endregion
		}
	}
}