using System;
using System.Diagnostics;
using System.Text;

// Creation date: 22.04.2002
// Checked: xx.05.2002
// Author: Otto Mayer (mot@root.ch)
// Version: 1.01

// Report.NET copyright 2002-2004 root-software ag, Brglen Switzerland - O. Mayer, S. Spirig, R. Gartenmann, all rights reserved
// This library is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation, version 2.1 of the License.
// This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details. You
// should have received a copy of the GNU Lesser General Public License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA www.opensource.org/licenses/lgpl-license.html

namespace Root.Reports {
  /// <summary></summary>
  public sealed class PdfFontPropData : FontPropData {
    /// <summary>ID of the PDF object</summary>
    internal Int32 iObjId;

    internal readonly String sKey;

    internal readonly Type1FontData afm;

    /// <summary>This variable allows a quick test, whether the font properties are registered for the current page.
    /// If <c>pdfPageData_Registered</c> contains the current page, then it has been registered before.</summary>
    private PdfPageData _pdfPageData_Registered;

    //----------------------------------------------------------------------------------------------------x
    /// <summary></summary>
    /// <param name="sKey"></param>
    /// <param name="afm"></param>
    public PdfFontPropData(String sKey, Type1FontData afm) {
      this.sKey = sKey;
      this.afm = afm;
      Init();
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>This variable allows a quick test, whether the font properties are registered for the current page.
    /// If <c>pdfPageData_Registered</c> contains the current page, then it has been registered before.</summary>
    internal PdfPageData pdfPageData_Registered {
      get { return _pdfPageData_Registered; }
      set {
        #if (DEBUG)
        StackFrame sf = new StackFrame(1);
        Debug.Assert(sf.GetMethod().Name == "RegisterFontProp");
        #endif
        _pdfPageData_Registered = value;
      }
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Gets the size factor of the font.</summary>
    /// <param name="fontProp">Font properties</param>
    /// <returns>Size factor</returns>
    internal Double rSizeFactor(FontProp fontProp) {
      if (fontProp.fontDef.sFontName == "Symbol" || fontProp.fontDef.sFontName == "ZapfDingbats") {
        Type1FontData.CharMetrics afmCharMetrics = afm.afmCharMetrics(74);
        Debug.Assert(!(Single.IsNaN(afmCharMetrics.fBBox_ury)));
        Debug.Assert(!(Single.IsNaN(afmCharMetrics.fBBox_lly)));
        return fontProp.rSize * 1000.0 / (afmCharMetrics.fBBox_ury - afmCharMetrics.fBBox_lly);
      }
      Debug.Assert(!(Single.IsNaN(afm.fCapHeight)));
      return fontProp.rSize * 1000.0 / afm.fCapHeight;
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary></summary>
    internal protected override void Init() {
      iObjId = 0;
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Gets the height of the font in 1/72 inches.</summary>
    /// <param name="fontProp">Font properties</param>
    /// <returns>Height of the font in 1/72 inches</returns>
    internal  protected override Double rHeight(FontProp fontProp) {
      Single fCapHeight = afm.fCapHeight;
      if (Single.IsNaN(fCapHeight)) {
        fCapHeight = afm.fFontBBox_ury - afm.fFontBBox_lly;
      }
      Debug.Assert(!(Single.IsNaN(fCapHeight)));
      return rSizeFactor(fontProp) * fCapHeight / 1000F;
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Gets the width of the specified text with this font in 1/72 inches.</summary>
    /// <param name="fontProp">Font properties</param>
    /// <param name="sText">Text</param>
    /// <returns>Width of the text in 1/72 inches</returns>
    internal  protected override Double rWidth(FontProp fontProp, String sText) {
      return rSizeFactor(fontProp) * afm.rWidth(sText) / 1000;
    }

  }
}
