using System;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Text;
using System.Drawing.Imaging;

// Creation date: 22.04.2002
// Checked: xx.05.2002
// Author: Otto Mayer (mot@root.ch)
// Version: 1.01

// Report.NET copyright 2002-2004 root-software ag, Brglen Switzerland - O. Mayer, S. Spirig, R. Gartenmann, all rights reserved
// This library is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation, version 2.1 of the License.
// This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details. You
// should have received a copy of the GNU Lesser General Public License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA www.opensource.org/licenses/lgpl-license.html

namespace Root.Reports {
  /// <summary>PDF Page Layout</summary>
  /// <remarks>This attribute specifies the page layout to be used when the document is opened.</remarks>
  public enum PageLayout {
    /// <summary>Display one page at a time.</summary>
    SinglePage, 
    /// <summary>Display the pages in one column.</summary>
    OneColumn,
    /// <summary>Display the pages in two columns, with oddnumbered pages on the left.</summary>
    TwoColumnLeft,
    /// <summary>Display the pages in two columns, with oddnumbered pages on the right.</summary>
    TwoColumnRight
  }

  /// <summary>PDF Page Layout</summary>
  public enum PageMode {
    /// <summary>Standard</summary>
    UseNone,
    /// <summary>Outlines</summary>
    UseOutlines,
    /// <summary>Thumbs</summary>
    UseThumbs,
    /// <summary>Full screen</summary>
    FullScreen
  }

  /// <summary>PDF Page Layout</summary>
  public enum NonFullScreenPageMode {
    /// <summary>Standard</summary>
    UseNone,
    /// <summary>Outlines</summary>
    UseOutlines,
    /// <summary>Thumbs</summary>
    UseThumbs
  }
  //public enum Trapped { True, False, Unknown }

  /// <summary>PDF-Formatter</summary>
  public /*sealed !!!*/ class PdfFormatter : Formatter {
    /// <summary>Report</summary>
    private Report report;

    // <summary>PDF output stream</summary>
    //private Stream stream;

    /// <summary>number of bytes written to the PDF output stream</summary>
    private Int32 iBytesWrittenToStream;

    /// <summary>PDF output stream for ASCII text</summary>
    private BufferedStream bufferedStream;
 
    /// <summary>output buffer</summary>
    private StringBuilder sb;

    /// <summary>Array that holds the object position of each PDF object</summary>
    private ArrayList al_ObjPos;

    /// <summary>List of all fonts that must be defined in the PDF document</summary>
    Hashtable ht_FontProp;

    /// <summary>List of all AFM font definitions that are defined for the document</summary>
    Hashtable ht_Afm;

    /// <summary>id of the catalog object</summary>
    private Int32 iObjBaseCatalog;
    /// <summary>id of the metadata object</summary>
    private Int32 iObjMetadata;
    /// <summary>id of the viewer preferences object</summary>
    private Int32 iObjViewerPreferences;
    /// <summary>base id of the font objects</summary>
    private Int32 iObjBaseFont;
    /// <summary>id of the pages object</summary>
    private Int32 iObjBasePages;
    /// <summary>base id of the contents objects</summary>
    private Int32 iObjBaseContents;

    /// <summary>position of the xref table</summary>
    private Int32 iXRefPos;

    /// <summary>Enumeration of all default PDF fonts</summary>
    /*
    private enum DefaultFont {
      Helvetica, Helvetica_Bold, Helvetica_Oblique, Helvetica_BoldOblique,
      Times_Roman, Times_Bold, Times_Italic, Times_BoldItalic,
      Courier, Courier_Bold, Courier_Oblique, Courier_BoldOblique,
      Symbol, ZapfDingbats
    }
    */

    private const String sFontAbbr = "F";

    //----------------------------------------------------------------------------------------------------x

    /// <summary>Title of the document</summary>
    public String sTitle;

    /// <summary>The name of the person who created the document</summary>
    public String sAuthor;

    /// <summary>Subject of the document</summary>
    public String sSubject;

    /// <summary>Keywords associated with the document</summary>
    public String sKeywords;

    /// <summary>Application that created the document</summary>
    public String sCreator;

    /// <summary>Program that converted the document to PDF</summary>
    private String sProducer = "Report.NET by root-software ag";

    /// <summary>Creation date and time of  the document</summary>
    public DateTime dt_CreationDate;

    /// <summary>Modification date and time of  the document</summary>
    private DateTime dt_ModDate = DateTime.Now;

    //public Trapped bTrapped = Trapped.Unknown;

    /// <summary>Determines the page layout in the PDF document</summary>
    public PageLayout pageLayout = PageLayout.SinglePage;

    /// <summary>Determines the page Mode in the PDF document</summary>
    public PageMode pageMode = PageMode.UseNone;

    /// <summary>Hide toolbar</summary>
    public Boolean bHideToolBar = false;

    /// <summary>Hide menu bar</summary>
    public Boolean bHideMenubar = false;

    /// <summary>Hide window UI</summary>
    public Boolean bHideWindowUI = false;

    /// <summary>Fit window</summary>
    public Boolean bFitWindow = false;

    /// <summary>Center window</summary>
    public Boolean bCenterWindow = false;

    /// <summary>Display document title</summary>
    public Boolean bDisplayDocTitle = false;

    /// <summary>Full screen page mode</summary>
    public NonFullScreenPageMode nonFullScreenPageMode = NonFullScreenPageMode.UseNone;

    /// <summary>Open action URI</summary>
    public String sOpenActionURI = null;

    /// <summary>Open action launch</summary>
    public String sOpenActionLaunch = null;

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Current pen width</summary>
    private Double rPenWidth_Cur;

    private Double rPatternOn;
    private Double rPatternOff;

    private Boolean bColor_rg;
    private Color color_rg;

    private Boolean bColor_RG;
    private Color color_RG;
 
    private String sFont_Cur;

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Initializes a new instance of the PDF formatter class.</summary>
    /// <example>
    /// <code>
    /// using Root.Report;
    /// using System;
    /// namespace ReportSample {
    /// class PdfPropertiesSample : Report {
    ///   public static void Main() {
    ///     PdfFormatter pf = new PdfFormatter();
    ///     pf.sTitle = "PDF Sample";
    ///     pf.sAuthor = "Otto Mayer, mot@root.ch";
    ///     pf.sSubject = "Sample of some PDF features";
    ///     pf.sKeywords = "Sample PDF Report.NET";
    ///     pf.sCreator = "Report.NET Sample Application";
    ///     pf.dt_CreationDate = new DateTime(2002, 8, 15, 0,0,0,0);
    ///     pf.pageLayout = PageLayout.TwoColumnLeft;
    ///     pf.bHideToolBar = true;
    ///     pf.bHideMenubar = false;
    ///     pf.bHideWindowUI = true;
    ///     pf.bFitWindow = true;
    ///     pf.bCenterWindow = true;
    ///     pf.bDisplayDocTitle = true;
    /// 
    ///     RT.ViewPDF(new PdfPropertiesSample(pf), "PdfPropertiesSample.pdf");
    ///   }
    ///
    ///   public PdfPropertiesSample(Formatter formatter) : base(formatter) {
    ///   }
    ///
    ///   protected override void Create() {
    ///     FontDef fd = new FontDef(this, FontDef.StandardFont.Helvetica);
    ///     FontProp fp = new FontPropMM(fd, 4);
    ///     FontProp fp_Title = new FontPropMM(fd, 11);
    ///     fp_Title.bBold = true;
    ///
    ///     Page page = new Page(this);
    ///     page.AddCenteredMM(40, new RepString(fp_Title, "PDF Properties Sample"));
    ///     fp_Title.rSizeMM = 8;
    ///     page.AddCenteredMM(100, new RepString(fp_Title, "First Page"));
    ///     page.AddCenteredMM(120, new RepString(fp, "Choose &lt;Document Properties, Summary&gt; from the"));
    ///     page.AddCenteredMM(126, new RepString(fp, "File menu to display the document properties"));
    ///
    ///     page = new Page(this);
    ///     page.AddCenteredMM(100, new RepString(fp_Title, "Second Page"));
    ///   }
    /// }
    /// }
    /// </code>
    /// </example>
    public PdfFormatter() {
      ht_Afm = new Hashtable(20);
      ht_FontProp = new Hashtable(50);
    }

    //----------------------------------------------------------------------------------------------------
    /// <summary>Builds the catalog object.</summary>
    /// <param name="fontProp"></param>
    internal Type1FontData afm_Register(FontProp fontProp) {
      String sFontName = fontProp.fontDef.sFontName;
      if (sFontName == "Arial") {
        sFontName = "Helvetica";
      }
      if (fontProp.bBold) {
        if (sFontName == "Times-Roman") {
          sFontName = "Times-Bold";
        }
        else {
          sFontName += "-Bold";
        }
      }
      if (fontProp.bItalic) {
        if (sFontName == "Courier" || sFontName == "Helvetica") {
          sFontName += "-Oblique";
        }
        else if (sFontName == "Courier-Bold" || sFontName == "Helvetica-Bold") {
          sFontName += "Oblique";
        }
        else if (sFontName == "Times-Roman") {
          sFontName = "Times-Italic";
        }
        else {
          sFontName += "Italic";
        }
      }
      Type1FontData afm = (Type1FontData)ht_Afm[sFontName];
      if (afm == null) {
        Stream stream = GetType().Assembly.GetManifestResourceStream("Root.Reports.PDF.afm." + sFontName + ".afm");
        afm = new Type1FontData(stream, FontData.Style.Standard);
        //afm = new Afm(AppDomain.CurrentDomain.SetupInformation.ApplicationBase + @"\PDF\afm\" + sFontName + ".afm");
      }
      return afm;
    }

    //----------------------------------------------------------------------------------------------------
    /// <summary>Builds the catalog object.</summary>
    private void BuildCatalog() {
      NewObjId();
      Debug.Assert(iObjIdCur == iObjBaseCatalog, "invalid object id of catalog");
      WriteLine(iObjBaseCatalog + " 0 obj");
      WriteLine("<<");
      WriteLine("/Type /Catalog");
      WriteLine("/Version /1.4");
      // page 83 3.6.1 !!!
      WriteLine("/Pages " + iObjBasePages + " 0 R");
      if (pageLayout != PageLayout.SinglePage) {
        WriteLine("/PageLayout /" + Enum.GetName(typeof(PageLayout), pageLayout));
      }
      if (pageMode != PageMode.UseNone) {
        WriteLine("/PageMode /" + Enum.GetName(typeof(PageMode), pageMode));
      }
      WriteLine("/ViewerPreferences " + iObjViewerPreferences + " 0 R");
      if (sOpenActionURI != null){
        WriteLine("/OpenAction <</Type /Action");
        WriteLine("/S /URI");
        WriteLine("/URI (" + sOpenActionURI + ")");
        WriteLine(">>");
      } else if (sOpenActionLaunch != null ) {
        WriteLine("/OpenAction <</Type /Action");
        WriteLine("/S /Launch");
        WriteLine("/F (" + sOpenActionLaunch + ") ");
        WriteLine(">>");
      }

      WriteLine(">>");
      WriteLine("endobj");

      // meta data     
      NewObjId();
      Debug.Assert(iObjIdCur == iObjMetadata, "invalid object id of metadata object");
      WriteLine(iObjMetadata + " 0 obj");
      WriteLine("<<");
      if (sTitle != null) {
        WriteLine("/Title " + RT.sPdfString(sTitle));
      }
      if (sAuthor != null) {
        WriteLine("/Author " + RT.sPdfString(sAuthor));
      }
      if (sSubject != null) {
        WriteLine("/Subject " + RT.sPdfString(sSubject));
      }
      if (sKeywords != null) {
        WriteLine("/Keywords " + RT.sPdfString(sKeywords));
      }
      if (sCreator != null) {
        WriteLine("/Creator " + RT.sPdfString(sCreator));
      }
      if (sProducer != null) {
        WriteLine("/Producer " + RT.sPdfString(sProducer));
      }
      if (dt_CreationDate.Year > 0) {
        String sMonth = dt_CreationDate.Month.ToString();
        if (sMonth.Length < 2){
          sMonth = "0" + sMonth;
        }
        String sDay = dt_CreationDate.Day.ToString();
        if (sDay.Length < 2){
          sDay = "0" + sDay;
        }
        String sTemp = dt_CreationDate.Year.ToString() + sMonth + sDay;
        WriteLine("/CreationDate " + RT.sPdfString(sTemp));
      }
      if (dt_ModDate.Year > 0) {
        String sMonth = dt_ModDate.Month.ToString();
        if (sMonth.Length < 2){
          sMonth = "0" + sMonth;
        }
        String sDay = dt_ModDate.Day.ToString();
        if (sDay.Length < 2){
          sDay = "0" + sDay;
        }
        String sTemp = dt_ModDate.Year.ToString() + sMonth + sDay;
        WriteLine("/ModDate " + RT.sPdfString(sTemp));
      }
      //if (bTrapped != Trapped.Unknown) {
      //WriteLine("/Trapped /" + Enum.GetName(typeof(Trapped), trapped));
      //}
      WriteLine(">>");
      WriteLine("endobj");

      // viewer preferences
      NewObjId();
      Debug.Assert(iObjIdCur == iObjViewerPreferences, "invalid object id of viewer preferences object");
      WriteLine(iObjViewerPreferences + " 0 obj");
      WriteLine("<<");
      if (bHideToolBar) {
        WriteLine("/HideToolbar  true");
      }
      if (bHideMenubar) {
        WriteLine("/HideMenubar  true");
      }
      if (bHideWindowUI) {
        WriteLine("/HideWindowUI  true");
      }
      if (bFitWindow) {
        WriteLine("/FitWindow  true");
      }
      if (bCenterWindow) {
        WriteLine("/CenterWindow  true");
      }
      if (bDisplayDocTitle) {
        WriteLine("/DisplayDocTitle  true");
      }
      if (nonFullScreenPageMode != NonFullScreenPageMode.UseNone) {
        WriteLine("/NonFullScreenPageMode /" + Enum.GetName(typeof(NonFullScreenPageMode), nonFullScreenPageMode));
      }
      WriteLine(">>");
      WriteLine("endobj");
      /*
            NewObjId();
            Debug.Assert(iObjIdCur == iObjEncoding, "invalid object id of viewer preferences object");
            WriteLine(iObjEncoding + " 0 obj");
            WriteLine("<<");
            WriteLine("/Type /Encoding");
            WriteLine("/Differences [");
            WriteLine("065 /copyright");
            WriteLine("]");
            WriteLine(">>");
            WriteLine("endobj");
            */
      
    }
    //----------------------------------------------------------------------------------------------------
    /// <summary>Builds the font objects.</summary>
    private void BuildFontObjects() {
      Debug.Assert(iObjIdCur + 1 == iObjBaseFont, "invalid object id of font objects");
      StringBuilder sb_Error = null;
      SortedList sl_FontProp = new SortedList();
      foreach (FontProp fontProp in ht_FontProp.Values) {
        PdfFontPropData pdfFontPropData = (PdfFontPropData)fontProp.fontPropData;
        sl_FontProp.Add(pdfFontPropData.iObjId, fontProp);
      }

      foreach (FontProp fontProp in sl_FontProp.Values) {
        PdfFontPropData pdfFontPropData = (PdfFontPropData)fontProp.fontPropData;
        String sFontName = pdfFontPropData.afm.sFontName;
        if (true /*Enum.IsDefined(typeof(DefaultFont), sFontName)*/) {
          NewObjId();
          Debug.Assert(iObjIdCur == pdfFontPropData.iObjId, "invalid object index of font object");
          WriteLine(pdfFontPropData.iObjId + " 0 obj");
          WriteLine("<<");
          WriteLine("/Type /Font");
          WriteLine("/Subtype /Type1");
          WriteLine("/Name /" + sFontAbbr + pdfFontPropData.iObjId);
          //WriteLine("/Encoding " + iObjEncoding + " 0 R");
          WriteLine("/BaseFont /" + sFontName);
          if (pdfFontPropData.afm.sFamilyName != "ZapfDingbats" && pdfFontPropData.afm.sFamilyName != "Symbol") {
            WriteLine("/Encoding /WinAnsiEncoding");
          }
          WriteLine(">>");
          WriteLine("endobj");
        }
#if (xy)
        else {
          if (sb_Error == null) {
            sb_Error = new StringBuilder(100);
          }
          else {
            sb_Error.Append(", ");
          }
          sb_Error.Append(sFontName);
        }
#endif
      }
      if (sb_Error != null) {
        throw new ReportException("unknown fonts:" + sb_Error.ToString());
      }
    }

    //----------------------------------------------------------------------------------------------------
    /// <summary>Builds the contents objects of the page.</summary>
    private void BuildPage(Page page) {
      NewObjId();
      WriteLine(iObjIdCur + " 0 obj");
      Write("<</Length ");
      FlushBuffer();

      rPenWidth_Cur = -1;
      rPatternOn = -1;
      rPatternOff = -1;
      bColor_rg = false;
      bColor_RG = false;
      sFont_Cur = null;

      BuildPageFromContainer(page);
      WriteDirect(sb.Length + ">>\nstream\n");
      WriteLine("endstream");
      WriteLine("endobj");
      WriteLine(">>");
    }

    //----------------------------------------------------------------------------------------------------
    /// <summary>Prepares the PDF-object structure for a container.</summary>
    private void BuildPageFromContainer(Container container) {
      Page page = container.page;

      foreach (RepObj repObj in container) {
        MatrixD m = container.matrixD.Clone();
        m.Multiply(repObj.matrixD);
        Boolean bComplex = m.bComplex;

        if (repObj is Container) {
          BuildPageFromContainer((Container)repObj);
        }
        else if (repObj is RepArcBase) {  /* arc */
          RepArcBase arc = (RepArcBase)repObj;
          Double rOfsX = repObj.rWidth * (-repObj.rAlignH + 0.5);
          Double rOfsY = repObj.rHeight * (1 - repObj.rAlignV - 0.5);
          m.Multiply(new MatrixD(1, 0, 0, 1, rOfsX, rOfsY));

          String sDrawCommand = null;
          if (arc._penProp != null && arc._penProp.rWidth != 0.0) {
            Write_Pen(arc._penProp);
            if (arc._brushProp != null) {
              Write_Brush(arc._brushProp);
              sDrawCommand = "b";  // close, fill and stroke path
            }
            else {
              sDrawCommand = (repObj is RepArc ? "S" : "s");  // stroke : close and stroke path
            }
          }
          else if (arc._brushProp != null) {
            Write_Brush(arc._brushProp);
            sDrawCommand = "f";  // fill path
          }
          else {
            return;
          }

          Double rA = arc.rWidth / 2;
          Double rA2 = rA * rA;
          Double rB = arc.rHeight / 2;
          Double rB2 = rB * rB;
          
          // start point: P0
          Double rAngle0 = RT.rRadianFromDegree(arc._rStartAngle);
          Double rP0X, rP0Y;
          arc.GetEllipseXY(rAngle0, out rP0X, out rP0Y);
          WriteLine("q");
          WriteLine(sMatrix(page, m) + " cm");
          if (repObj is RepArc || repObj is RepCircle || repObj is RepEllipse) {
            WriteLine(RT.sPdfDim(rP0X) + " " + RT.sPdfDim(rP0Y) + " m");
          }
          else {
            WriteLine("0 0 m");
            WriteLine(RT.sPdfDim(rP0X) + " " + RT.sPdfDim(rP0Y) + " l");
          }

          Double r = arc._rSweepAngle / 180 * Math.PI;
          Int32 iNumberOfArcs = ((Int32)(r / (Math.PI / 3.0))) + 1;
          Double rSweepAngle = r / iNumberOfArcs;
          for (Int32 iArc = 0;  iArc < iNumberOfArcs;  iArc++) {
            // end point: P3
            Double rAngle3 = rAngle0 + rSweepAngle;
            Double rP3X, rP3Y;
            arc.GetEllipseXY(rAngle3, out rP3X, out rP3Y);

            Double rAngle05 = rAngle0 + rSweepAngle / 2.0;
            Double rMX, rMY;
            arc.GetEllipseXY(rAngle05, out rMX, out rMY);
            
            
            Double rP1X, rP2X, rP1Y, rP2Y;
            Double rDenominator = rP0X * rP3Y - rP3X * rP0Y;
            Debug.Assert(!RT.bEquals(rDenominator, 0, 0.0001), "parallel tangents never appears if the sweep angle is less than PI/2");
            if (RT.bEquals(rP0Y, 0, 0.0001)) {
              Debug.Assert(!RT.bEquals(rP3Y, 0, 0.0001), "P0 and P3 on x-axis: never appears if the sweep angle is less than PI/2");
              rP1X = rP0X;
              rP2X = 8.0 / 3.0 * rMX -  4.0 / 3.0 * rP0X - rP3X / 3.0;
              rP1Y = 8.0 / 3.0 * rMY - rB2 / rP3Y + rB2 * rP3X * (8.0 * rMX - 4 * rP0X - rP3X) / (3.0 * rA2 * rP3Y) - rP3Y / 3.0;
              rP2Y = rB2 / rP3Y * (1 - rP2X * rP3X / rA2);
            }
            else if (RT.bEquals(rP3Y, 0, 0.0001)) {
              rP1X = 8.0 / 3.0 * rMX -  rP0X / 3.0 - 4.0 / 3.0 * rP3X;
              rP2X = rP3X;
              rP1Y = rB2 / rP0Y * (1 - rP0X * rP1X / rA2);
              rP2Y = 8.0 / 3.0 * rMY - rP0Y / 3.0 - rB2 / rP0Y + rB2 * rP0X * (8.0 * rMX - rP0X - 4 * rP3X) / (3.0 * rA2 * rP0Y);
            }
            else {
              rP1X = (3.0 * rA2 * rB2 * (rP0Y + rP3Y)
                    + rA2 * rP0Y * rP3Y * (rP0Y + rP3Y - 8 * rMY)
                    + rB2 * rP3X * rP0Y * (rP0X + rP3X - 8 * rMX))
                    / (3 * rB2 * rDenominator);
              rP2X = 8.0 / 3.0 * rMX - (rP0X + rP3X) / 3.0 - rP1X;

              rP1Y = rB2 / rP0Y * (1 - rP0X * rP1X / rA2);
              rP2Y = rB2 / rP3Y * (1 - rP2X * rP3X / rA2);
            }
            Debug.Assert(RT.bEquals(rMX, rP0X / 8.0 + 3.0 / 8.0 * rP1X + 3.0 / 8.0 * rP2X + rP3X / 8.0, 0.0001));
            Debug.Assert(RT.bEquals(rMY, rP0Y / 8.0 + 3.0 / 8.0 * rP1Y + 3.0 / 8.0 * rP2Y + rP3Y / 8.0, 0.0001));

            WriteLine(RT.sPdfDim(rP1X) + " " + RT.sPdfDim(rP1Y) + " " +
              RT.sPdfDim(rP2X) + " " + RT.sPdfDim(rP2Y) + " " +
              RT.sPdfDim(rP3X) + " " + RT.sPdfDim(rP3Y) + " c");

            rAngle0 += rSweepAngle;
            rP0X = rP3X;
            rP0Y = rP3Y;
          }
          WriteLine(sDrawCommand);
          WriteLine("Q");
        }
        else if (repObj is RepImage) {  /* image */
          RepImage repImage = (RepImage)repObj;
          Double rOfsX = repObj.rWidth * repObj.rAlignH;
          Double rOfsY = repObj.rHeight * (1 - repObj.rAlignV);
          m.Multiply(new MatrixD(1, 0, 0, 1, -rOfsX, rOfsY));
          m.Scale(repImage.rWidth, repImage.rHeight);
          WriteLine("q");
          WriteLine(sMatrix(page, m) + " cm");
          WriteLine("/I" + (Int32)repImage.imageData.object_Data + " Do");
          WriteLine("Q");
        }
        else if (repObj is RepRect) {  /* rectangle */
          RepRect repRect = (RepRect)repObj;
          Double rOfsX = repObj.rWidth * repObj.rAlignH;
          Double rOfsY = repObj.rHeight * (1 - repObj.rAlignV);
          m.Multiply(new MatrixD(1, 0, 0, 1, -rOfsX, rOfsY));
          if (repRect.penProp != null) {
            if (repRect.penProp.rWidth > 0f) {
              Write_Pen(repRect.penProp);
            }
            else {
              repRect.penProp = null;
            }
          }
          if (repRect.brushProp != null) {
            Write_Brush(repRect.brushProp);
          }

          if (bComplex) {
            WriteLine("q");
            WriteLine(sMatrix(page, m) + " cm");
            WriteLine(RT.sPdfDim(0) + " " + RT.sPdfDim(0) + " " + RT.sPdfDim(repRect.rWidth) + " " + RT.sPdfDim(repRect.rHeight) + " re");
            WriteLine(repRect.penProp == null ? "f" : (repRect.brushProp == null ? "S" : "B"));
            WriteLine("Q");
          }
          else {
            WriteLine(sPoint(page, m.rDX, m.rDY) + " " + RT.sPdfDim(repRect.rWidth) + " " + RT.sPdfDim(repRect.rHeight) + " re");
            WriteLine(repRect.penProp == null ? "f" : (repRect.brushProp == null ? "S" : "B"));
          }
        }
        else if (repObj is RepLine) {  /* line */
          RepLine repLine = (RepLine)repObj;
          Double rOfsX = repObj.rWidth * repObj.rAlignH;
          Double rOfsY = repObj.rHeight * repObj.rAlignV;
          m.Multiply(1, 0, 0, 1, -rOfsX, -rOfsY);
          if (repLine.penProp.rWidth > 0f) {
            Write_Pen(repLine.penProp);
            WriteLine(sPoint(page, m.rDX, m.rDY) + " m");
            WriteLine(sPoint(page, m, repLine.rWidth, repLine.rHeight) + " l");
            WriteLine("S");
          }
        }
        else if (repObj is RepString) {
          RepString repString = (RepString)repObj;
          Double rOfsX = repString.fontProp.rWidth(repString.sText) * repObj.rAlignH;
          Double rOfsY = repString.fontProp.rHeight() * (1 - repObj.rAlignV);
          m.Multiply(new MatrixD(1, 0, 0, 1, -rOfsX, rOfsY));

          WriteLine("BT");
          Write_Font(repString.fontProp);
          if (bComplex) {
            WriteLine(sMatrix(page, m) + " Tm");
          }
          else {
            WriteLine(sPoint(page, m.rDX, m.rDY) + " Td");
          }
          Write("(");  // !!!
          String s = repString.sText;
          for (Int32 i = 0;  i < s.Length;  i++) {
            Int32 iChar = (Int32)s[i];
            if (iChar < 256) {
              if (iChar == (Int32)'(' || iChar ==  (Int32)')' || iChar ==  (Int32)'\\') {
                sb.Append('\\');
              }
              sb.Append((Char)iChar);
            }
            else {
              if (iChar == 8364) {
                sb.Append("\\200");
              }
              //String sx = "\\" + (iChar / 64).ToString() + ((iChar / 8) % 8).ToString() + (iChar % 8).ToString();
              //sb.Append("\\" + (iChar / 64).ToString() + ((iChar / 8) % 8).ToString() + (iChar % 8).ToString());
            }
          }
          WriteLine(") Tj");
          WriteLine("ET");

          if (repString.fontProp.bUnderline) {
            PdfFontPropData pdfFontPropData = (PdfFontPropData)repString.fontProp.fontPropData;
            Double rSizeFactor = pdfFontPropData.rSizeFactor(repString.fontProp);
            PenProp pp = new PenProp(report, rSizeFactor * pdfFontPropData.afm.fUnderlineThickness / 1000, repString.fontProp.color);
            Write_Pen(pp);
            Double rD = rSizeFactor * pdfFontPropData.afm.fUnderlinePosition / 1000;
            WriteLine(sPoint(page, m, 0, -rD) + " m");
            Double rWidth = repString.fontProp.rWidth(repString.sText);
            WriteLine(sPoint(page, m, rWidth, -rD) + " l");
            WriteLine("S");
          }
        }
        else {
          throw new ReportException("unknown report object type [" + repObj.GetType().FullName + "]"); 
        }
        //WriteLine("Q");
      }
    }

    //----------------------------------------------------------------------------------------------------
    /// <summary>Builds the pages object.</summary>
    private void BuildPages() {
      NewObjId();
      Debug.Assert(iObjIdCur == iObjBasePages, "invalid object index of pages object");
      WriteLine(iObjBasePages + " 0 obj");
      WriteLine("<<");
      WriteLine("/Type /Pages");
      WriteLine("/Count " + report.iPageCount);
      WriteLine("/Kids [");
      foreach (Page page in report.enum_Page) {
        PdfPageData pdfPageData = (PdfPageData)page.pageData;
        Write(pdfPageData.iObjId + " 0 R ");
      }
      WriteLine("]");
      WriteLine(">>");
      WriteLine("endobj");
    }

    //----------------------------------------------------------------------------------------------------
    /// <summary>Builds the contents objects of the pages.</summary>
    private void BuildPageContents() {
      Debug.Assert(iObjIdCur + 1 == iObjBaseContents, "invalid object index of page contents object");
      foreach (Page page in report.enum_Page) {
        PdfPageData pdfPageData = (PdfPageData)page.pageData;
        NewObjId();
        Debug.Assert(iObjIdCur == pdfPageData.iObjId, "invalid object index of page object");
        WriteLine(iObjIdCur + " 0 obj");
        WriteLine("<<");
        WriteLine("/Type /Page");
        WriteLine("/Parent " + iObjBasePages + " 0 R");

        WriteLine("/Resources <<");
        Write("/ProcSet [");
        if (pdfPageData.bProcSet_PDF) {
          Write("/PDF ");
        }
        if (pdfPageData.bProcSet_Text) {
          Write("/Text ");
        }
        if (pdfPageData.bProcSet_ImageB) {
          Write("/ImageB ");
        }
        if (pdfPageData.bProcSet_ImageC) {
          Write("/ImageC ");
        }
        //Added By TechnoGuru - jjborie@yahoo.fr - http://www.borie.org/
        if (pdfPageData.bProcSet_ImageI) {
          Write("/ImageI ");
        }
        WriteLine("]");

        WriteLine("/Font <<");
        foreach (FontProp fontProp in pdfPageData.icollection_FontProp) {
          PdfFontPropData pdfFontPropData = (PdfFontPropData)fontProp.fontPropData;
          WriteLine("/" + sFontAbbr + pdfFontPropData.iObjId + " " + pdfFontPropData.iObjId + " 0 R");
        }
        WriteLine(">>");

        if (report.ht_ImageData.Count > 0) {
          WriteLine("/XObject <<");
          foreach (ImageData imageData in report.ht_ImageData.Values) {
            WriteLine("/I" + (Int32)imageData.object_Data + " " + (Int32)imageData.object_Data + " 0 R");
          }
          WriteLine(">>");
        }
        /*
          if (hPdfWriter.m_pImageDataFirst != NULL) {
            hPdfWriter.Write("/XObject <<\n");
            for (CRsPdfWriter::CImageData* pImageData = hPdfWriter.m_pImageDataFirst;  pImageData != NULL;
                pImageData = pImageData->m_pImageDataNext) {
              hPdfWriter.Write(strRs "/Im" + strRsInt4(pImageData->m_nPdfObjId) + " " + strRsInt4(pImageData->m_nPdfObjId) + " 0 R\n");
            }
            hPdfWriter.Write(">>\n");
          }
        */
        /*-X-Objects-------------------------------------------------------------*/

        /*
    TestCheck(hPdfWriter.nGetPdfObjIdCur() + 1 == nObjXObjectFirst);

      hPdfWriter.NewPdfObj();
      hPdfWriter.Write(strRs
        strRsInt4(hPdfWriter.nGetPdfObjIdCur()) + " 0 obj\n"
        "<<\n"
        "/Type /XObject\n"
        "/Subtype /Image\n"
        "/Name /Im" + strRsInt4(pImageData->m_nPdfObjId) + "\n"
        "/Width "  + strRsInt4(hJpeg.dwGetWidth()) + "\n"
        "/Height "  + strRsInt4(hJpeg.dwGetHeight()) + "\n"
        "/Filter /DCTDecode\n"
        "/BitsPerComponent 8\n"
        "/ColorSpace /DeviceRGB\n"
        "/Length " + strRsInt4(hFile.GetLength()) + "\n"
        ">>\n"
        "stream\n"
      );

      char chBuf[16 * 1024];
      Int4 nRead;
      Loop {
        nRead = hFile.Read(chBuf, sizeof(chBuf));
        if (nRead == 0) {
          break;
        }
        hPdfWriter.Write(chBuf, nRead);
      }
      hFile.Close();

      hPdfWriter.Write(strRs
        "\n"
        "endstream\n" +
        "endobj\n"
      );
    }
      */
        WriteLine(">>");

        WriteLine("/MediaBox [0 0 " + RT.sPdfDim(page.rWidth) + " " + RT.sPdfDim(page.rHeight) + " ]");
        WriteLine("/CropBox [0 0 " + RT.sPdfDim(page.rWidth) + " " + RT.sPdfDim(page.rHeight) + " ]");
        WriteLine("/Rotate 0");
        WriteLine("/Contents " + (iObjIdCur + 1) + " 0 R");
        WriteLine(">>");
        WriteLine("endobj");

        BuildPage(page);
      }
    }

    //----------------------------------------------------------------------------------------------------
    /// <summary>Builds the trailer.</summary>
    private void BuildTrailer() {
      WriteLine("trailer");
      WriteLine("<<");
      WriteLine("/Size " + (iObjIdCur + 1));
      WriteLine("/Root " + iObjBaseCatalog + " 0 R");
      WriteLine("/Info " + (iObjBaseCatalog + 1) + " 0 R");
      WriteLine(">>");
      WriteLine("startxref");
      WriteLine(iXRefPos.ToString());
      Write("%%EOF");  // no need for EOL
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Builds the xref structure.</summary>
    private void BuildXRef() {
      FlushBuffer();
      iXRefPos = iBytesWrittenToStream;
      WriteLine("xref");
      WriteLine("0 " + (iObjIdCur + 1));
      Write("0000000000 65535 f\r\n");
      foreach (Int32 iOfs in al_ObjPos) {
        Write(iOfs.ToString("D10") + " 00000 n\r\n");
      }
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Builds the xref structure.</summary>
    private void BuildXObjects() {
//      foreach (ImageData imageData in report.ht_ImageData.Values) {
//        imageData.stream.Position = 0;
//        //Added By TechnoGuru - jjborie@yahoo.fr - http://www.borie.org/
//        using(Image image = Image.FromStream(imageData.stream)) {
//          // Tiff support
//          //Added By TechnoGuru - jjborie@yahoo.fr - http://www.borie.org/
//          if(image.RawFormat.Equals(ImageFormat.Tiff)) {
//          NewObjId();
//          Debug.Assert(iObjIdCur == (Int32)imageData.object_Data, "invalid object id of viewer preferences object");
//          WriteLine((Int32)imageData.object_Data + " 0 obj");
//          WriteLine("<<");
//          WriteLine("/Type /XObject");
//          WriteLine("/Subtype /Image");
//          WriteLine("/Name /I" + (Int32)imageData.object_Data);
//          WriteLine("/Width " + image.Width);
//          WriteLine("/Height " + image.Height);
//          //Int32 iImageStart = 0;
//          //Int32 iImageLength = 0;
//          //ImageFormat imageFormat = imageData.image.RawFormat;
//          //if (Object.Equals(imageFormat, ImageFormat.Jpeg)) {
//          //  iImageLength = (Int32)imageData.stream.Length;
//          //  WriteLine("/Filter /DCTDecode");
//          //}
//        else if (Object.Equals(imageFormat, ImageFormat.Tiff)) {
//          imageData.stream.Position = 0;
//          BinaryReader br = new BinaryReader(imageData.stream);
//          br.BaseStream.Position = 4;
//          Int32 iPointer = br.ReadInt32();
//          br.BaseStream.Position = iPointer;
//          Int16 nTagCount = br.ReadInt16();
//          Int32 iImage = 0;
//          Int32 iRows = 0;
//          Int32 iStripByteCounts = 0;
//          while (nTagCount > 0) {
//            Int16 nTagType = br.ReadInt16();
//            Int16 nDataType = br.ReadInt16();
//            Int32 iLength = br.ReadInt32();
//            Int32 iData = br.ReadInt32();
//            if (nTagType == 259) {  // Compression
//              if (iData == 1) {
//              }
//              else if (iData == 5) {
//                //WriteLine("/Filter /LZWDecode");
//                //WriteLine("/Filter /CCITTFaxDecode"); //changed
//              }
//              else {
//                Debug.Fail("unknown compression");
//              }
//            }
//            else if (nTagType == 262) {  // Photometric Interpretation
//              //Debug.Assert(iData == 2);  // RGB-Compression
//            }
//            else if (nTagType == 273) {  // Strip Offset
//              iImage = iData;
//              iRows = iLength;
//            }
//            else if (nTagType == 279) {  // Strip Byte Counts
//              iStripByteCounts = iData;
//            }
//            else if (nTagType == 317) {  // Predictor
//              Debug.Assert(iData == 2);  // Horizontal Predictor
//              //WriteLine("/DecodeParms [<< /Predictor 2 >>]");
//            }
//            nTagCount--;
//          }
//          iPointer = br.ReadInt32();
//
//          br.BaseStream.Position = iImage;
//          iImageStart = br.ReadInt32();
//          br.BaseStream.Position = iImage + (iRows - 1) * 4;
//          iImageLength = br.ReadInt32() - iImageStart;
//
//          br.BaseStream.Position = iStripByteCounts + (iRows - 1) * 4;
//          Int32 iStripLength = br.ReadInt32();
//          iImageLength += iStripLength;
//  
//          Console.WriteLine("");
//
//        }
//        else {
//          Debug.Fail("unknown image type: send image to mot@root.ch");
//        }
//        Int32 iBitsPerComponent = 0;
//        String sColorSpace = null;
//        if (imageData.image.PixelFormat == PixelFormat.Format8bppIndexed) {
//          iBitsPerComponent = 8;
//          sColorSpace = "DeviceGray";
//        }
//        else if (imageData.image.PixelFormat == PixelFormat.Format1bppIndexed) {
//          iBitsPerComponent = 1;
//          sColorSpace = "DeviceGray";
//        }
//        else if (imageData.image.PixelFormat == PixelFormat.Format24bppRgb) {
//          iBitsPerComponent = 8;
//          sColorSpace = "DeviceRGB";
//        }
//        else {
//          Debug.Fail("unknown image format: send image to mot@root.ch");
//        }
//        WriteLine("/BitsPerComponent " + iBitsPerComponent.ToString());
//        WriteLine("/ColorSpace /" + sColorSpace);
//        WriteLine("/Length " + iImageLength.ToString());
//        WriteLine(">>");
//        WriteLine("stream");
//        FlushBuffer();
//        
//        imageData.stream.Position = 0;
//        BinaryReader r = new BinaryReader(imageData.stream);
//        if (iImageStart > 0) {
//          r.BaseStream.Position = iImageStart;
//        }
//        Byte[] aByte = r.ReadBytes(iImageLength);
//        r.Close();
//
//        //stream.Flush();
//        bufferedStream.Write(aByte, 0, aByte.Length);
//        iBytesWrittenToStream += aByte.Length;
//        WriteLine("\nendstream");
//        WriteLine("endobj");
//      }
      foreach (ImageData imageData in report.ht_ImageData.Values) {
        imageData.stream.Position = 0;
        //Added By TechnoGuru - jjborie@yahoo.fr - http://www.borie.org/
        using(Image image = Image.FromStream(imageData.stream)) {
          // Tiff support
          //Added By TechnoGuru - jjborie@yahoo.fr - http://www.borie.org/
          if(image.RawFormat.Equals(ImageFormat.Tiff)) {
            NewObjId();
            Debug.Assert(iObjIdCur == (Int32)imageData.object_Data, "invalid object id of viewer preferences object"); 
            WriteLine((Int32)imageData.object_Data + " 0 obj"); 
            WriteLine("<<"); 
            WriteLine("/Type /XObject"); 
            WriteLine("/Subtype /Image"); 
            WriteLine("/Name /I" + (Int32)imageData.object_Data); 
            WriteLine("/Width " + image.Width); 
            WriteLine("/Height " + image.Height);

            // Handle B&W format with CCIT
            if(image.PixelFormat.Equals(PixelFormat.Format1bppIndexed)) {
              imageData.stream.Position = 0; 
              BinaryReader r = new BinaryReader(imageData.stream);
              //CCIT 4
              //This String contain the startinf sequence of the tiff file for CCIT 4
              string patternTiffFile = "";
              string startTiffFile = "";
              int index = 0;
              int i = 0;
              for(index=0; index < 2048; index++) {
                startTiffFile += (char) (((i = r.ReadByte()) == 0?1:i));
              }
              int startPositionTiff = startTiffFile.IndexOf(patternTiffFile);
              //CCIT 3
              if(startPositionTiff == -1) {
                //This String contain the startinf sequence of the tiff file for CCIT 3
                patternTiffFile = "                                                                             ";
                startPositionTiff = startTiffFile.IndexOf(patternTiffFile);
                WriteLine("/Filter [/CCITTFaxDecode]");
                WriteLine("/DecodeParms [<< /Columns " + image.Width + "  /Rows " + image.Height + " /EndOfBlock false/EncodedByteAlign true >>]"); // This line was added to support TIFF 
              }
              else {
                WriteLine("/Filter /CCITTFaxDecode");
                WriteLine("/DecodeParms << /K -1 /Columns " + image.Width + " >>"); // This line was added to support TIFF 
              }
              if(startPositionTiff == -1)startPositionTiff = 512;
              WriteLine("/BitsPerComponent 1");
              WriteLine("/ColorSpace /DeviceGray");
              Int64 lLength = imageData.stream.Length; 
              WriteLine("/Length " + (lLength - startPositionTiff).ToString()); // CHANGED 
              WriteLine(">>"); 
              WriteLine("stream"); 
              FlushBuffer(); 

              imageData.stream.Position = 0;
              r.ReadBytes((Int32)startPositionTiff);
              Byte[] aByte = r.ReadBytes((Int32)lLength - startPositionTiff); // CHANGED 
              r.Close(); 

              //stream.Flush(); 
              bufferedStream.Write(aByte, 0, aByte.Length); 
              iBytesWrittenToStream += aByte.Length; 
              WriteLine("\nendstream"); 
              WriteLine("endobj"); 
            }
            else if(image.PixelFormat.Equals(PixelFormat.Format4bppIndexed)) {
              //Not supported I don't have tiff file to test it =)

              //					  WriteLine("/Filter /CCITTFaxDecode");
              //					  WriteLine("/DecodeParms << /K -1 /Columns " + image.Width + " >>"); // This line was added to support TIFF 
              //					  WriteLine("/BitsPerComponent 4");
              //					  WriteLine("/ColorSpace /DeviceGray");
            }
            else {
              //Tiff in gray or color are converted as jpeg in the tiff
              MemoryStream ms = new MemoryStream();
              image.Save(ms,ImageFormat.Jpeg);
              Byte[] aByte = ms.GetBuffer();

              WriteLine("/BitsPerComponent 8");
              WriteLine("/ColorSpace /DeviceRGB");
              WriteLine("/Filter /DCTDecode");
              WriteLine("/Length " + (aByte.Length).ToString()); // CHANGED 
              WriteLine(">>"); 
              WriteLine("stream"); 
              FlushBuffer(); 

              bufferedStream.Write(aByte, 0, aByte.Length);
              iBytesWrittenToStream += aByte.Length;

              WriteLine("\nendstream"); 
              WriteLine("endobj"); 
            }
          }
          else {
            //JPEG format is the default one =)
            NewObjId();
            Debug.Assert(iObjIdCur == (Int32)imageData.object_Data, "invalid object id of viewer preferences object");
            WriteLine((Int32)imageData.object_Data + " 0 obj");
            WriteLine("<<");
            WriteLine("/Type /XObject");
            WriteLine("/Subtype /Image");
            WriteLine("/Name /I" + (Int32)imageData.object_Data);
            WriteLine("/Width " + image.Width);
            WriteLine("/Height " + image.Height);
            WriteLine("/Filter /DCTDecode");
            WriteLine("/BitsPerComponent 8");
            WriteLine("/ColorSpace /DeviceRGB");
            Int64 lLength = imageData.stream.Length;
            WriteLine("/Length " + lLength.ToString());
            WriteLine(">>");
            WriteLine("stream");
            FlushBuffer();
        
            //imageData.
            //System.IO.FileStream hFile = File.Open(imageData.sFileName, FileMode.Open, FileAccess.Read, FileShare.Read);
            imageData.stream.Position = 0;
            BinaryReader r = new BinaryReader(imageData.stream);
            Byte[] aByte = r.ReadBytes((Int32)lLength);
            r.Close();

            //stream.Flush();
            bufferedStream.Write(aByte, 0, aByte.Length);
            iBytesWrittenToStream += aByte.Length;
            WriteLine("\nendstream");
            WriteLine("endobj");
          }
          imageData.stream.Close();
          imageData.stream = null;
        }
      }
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Returns the id of the current PDF-object.</summary>
    private Int32 iObjIdCur {
      get { return al_ObjPos.Count; }
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary></summary>
    /// <param name="report"></param>
    /// <param name="stream"></param>
    public override void Create(Report report, Stream stream) {
      this.report = report;
      this.stream = stream;
      if (sTitle == null) {
        sTitle = report.sTitle;
      }
      if (sAuthor == null) {
        sAuthor = report.sAuthor;
      }

      foreach (FontProp fp in report.ht_FontProp.Values) {
        fp.fontPropData.Init();
      }

      //this.stream = stream;
      iBytesWrittenToStream = 0;
      sb = new StringBuilder(10000);
      al_ObjPos = new ArrayList(500);

      bufferedStream = new BufferedStream(stream);
      // Wichtig: Es muss die Standardeinstellung fr die Codierung verwendet werden !!!
      try {
        PrepareObjIds();

        sb.Length = 0;
        //sb.Remove(0, sb.Length);
        WriteLine("%PDF-1.4");  // Header
        FlushBuffer();
        BuildCatalog();
        BuildFontObjects();
        BuildPages();
        BuildPageContents();
        BuildXObjects();
        BuildXRef();
        BuildTrailer();
        FlushBuffer();
      }
      finally {
        bufferedStream.Flush();
        //bufferedStream.Close();
      }
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Copies the contents of the buffer to the PDF output stream.</summary>
    private void FlushBuffer() {
      Byte[] aByte = System.Text.Encoding.Default.GetBytes(sb.ToString());
      bufferedStream.Write(aByte, 0, aByte.Length);
      iBytesWrittenToStream += sb.Length;
      sb.Length = 0;
      //sb.Remove(0, sb.Length);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Creates a new PDF-object.</summary>
    private void NewObjId() {
      FlushBuffer();
      al_ObjPos.Add(iBytesWrittenToStream);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Creates a font properties data object for this formatter.</summary>
    /// <param name="fontProp"></param>
    internal protected override FontPropData fontPropData_CreateInstance(FontProp fontProp) {
      #if (DEBUG)
        StackFrame sf = new StackFrame(1);
        Debug.Assert(sf.GetMethod().Name == "get_fontPropData");
      #endif

      StringBuilder sb = new StringBuilder(50);
      sb.Append(fontProp.fontDef.sFontName);
      if (fontProp.bBold) {
        sb.Append(";B");
      }
      if (fontProp.bItalic) {
        sb.Append(";I");
      }
      String sKey = sb.ToString();

      FontProp fontProp_Reg = (FontProp)ht_FontProp[sKey];
      if (fontProp_Reg != null) {  // font registered
        //Debug.Assert(fontProp_Reg.fontPropData != null);
        return fontProp_Reg.fontPropData;
      }
      Type1FontData afm = afm_Register(fontProp);
      PdfFontPropData pdfFontPropData = new PdfFontPropData(sKey, afm);
      ht_FontProp.Add(sKey, fontProp);
      return pdfFontPropData;
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Creates a page data object for this formatter.</summary>
    internal protected override PageData pageData_CreateInstance() {
      return new PdfPageData();
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Initializes the font definition class.</summary>
    /// <param name="fontDef"></param>
    internal protected override void InitFontDef(FontDef fontDef) {
      String sStandard = fontDef.sFontName;
      String sBold = null;
      String sItalic = null;
      String sBoldItalic = null;
      if (sStandard == "Arial") {
        sStandard = "Helvetica";
      }
      Boolean bStandard = false;
      switch (fontDef.sFontName) {
        case "Courier": {
          bStandard = true;
          break;
        }
        case "Helvetica": {
          bStandard = true;
          break;
        }
        case "Symbol": {
          bStandard = false;
          break;
        }
        case "Times-Roman": {
          bStandard = false;
          sBold = "Times-Bold";
          sItalic = "Times-Italic";
          sBoldItalic = "Times-BoldItalic";
          break;
        }
        case "ZapfDingbats": {
          bStandard = false;
          break;
        }
      }
      if (bStandard) {
        sBold = sStandard + "-Bold";
        sItalic = sStandard + "-Oblique";
        sBoldItalic = sStandard + "-BoldOblique";
      }

      /*
      fontDef.fontData_Standard = new Type1FontData(sStandard, FontData.Style.Standard);
      if (sBold != null) {
        fontDef.fontData_Bold = new Type1FontData(sBold, FontData.Style.Bold);
      }
      if (sItalic != null) {
        fontDef.fontData_Italic = new Type1FontData(sItalic, FontData.Style.Italic);
      }
      if (sBoldItalic != null) {
        fontDef.fontData_BoldItalic = new Type1FontData(sBoldItalic, FontData.Style.BoldItalic);
      }
      */
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Prepares the PDF-object structure.</summary>
    private void PrepareObjIds() {
      Int32 iObjId = 1;
      iObjBaseCatalog = iObjId++;
      iObjMetadata = iObjId++;
      iObjViewerPreferences = iObjId++;
      //     iObjEncoding = iObjId++;

      // search all fonts and prepare the pages
      iObjBaseFont = iObjId;
      StringBuilder sb = new StringBuilder(50);

      foreach (Page page in report.enum_Page) {
        PrepareObjIdsForContainer((PdfPageData)page.pageData, ref iObjId, page);
      }
      // pages
      iObjBasePages = iObjId++;
      iObjBaseContents = iObjId;
      foreach (Page page in report.enum_Page) {
        PdfPageData pdfPageData = (PdfPageData)page.pageData;
        pdfPageData.iObjId = iObjId++;
        iObjId++;
      }
      foreach (ImageData imageData in report.ht_ImageData.Values) {
        imageData.object_Data = iObjId++;
      }
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Prepares the PDF-object structure for a container.</summary>
    /// <param name="pdfPageData"></param>
    /// <param name="iObjId"></param>
    /// <param name="container"></param>
    private void PrepareObjIdsForContainer(PdfPageData pdfPageData, ref Int32 iObjId, Container container) {
      foreach (RepObj repObj in container) {
        if (repObj is Container) {
          PrepareObjIdsForContainer(pdfPageData, ref iObjId, (Container)repObj);
        }
        else if (repObj is RepArcBase) {
          pdfPageData.bProcSet_PDF = true;
        }
        else if (repObj is RepImage) {
//          RepImage repImage = (RepImage)repObj;
//          ImageFormat imageFormat = repImage.imageData.image.RawFormat;
//          if (Object.Equals(imageFormat, ImageFormat.Jpeg)) {
//            pdfPageData.bProcSet_ImageC = true;
//          }
//          else if (Object.Equals(imageFormat, ImageFormat.Tiff)) {
//            pdfPageData.bProcSet_ImageB = true;
//          }
//          else {
//            Debug.Fail("unknown image type: send image to mot@root.ch");
//          }
          RepImage repImage = repObj as RepImage;
          repImage.imageData.stream.Position = 0;

          //Added By TechnoGuru - jjborie@yahoo.fr - http://www.borie.org/
          //To support tiff file
          //I reload Image from stream to be more scalable
          //(Dont't have like that lots of image object on memeory
          using (Image image = Image.FromStream(repImage.imageData.stream)) {
            if(image.RawFormat.Equals(ImageFormat.Tiff) && image.PixelFormat.Equals(PixelFormat.Format1bppIndexed)) {
              pdfPageData.bProcSet_ImageI = true; // CHANGED 
            }
            else if(image.RawFormat.Equals(ImageFormat.Tiff) || image.RawFormat.Equals(ImageFormat.Jpeg)) {
              pdfPageData.bProcSet_ImageC = true;
            }
            else {
              Debug.Fail("unknown image type: send image to mot@root.ch");
            }
          }
        }
        else if (repObj is RepLine) {
          pdfPageData.bProcSet_PDF = true;
        }
        else if (repObj is RepRect) {
          pdfPageData.bProcSet_PDF = true;
        }
        else if (repObj is RepString) {
          FontProp fontProp_String = ((RepString)repObj).fontProp;
          PdfFontPropData pdfFontPropData = (PdfFontPropData)fontProp_String.fontPropData;
          if (pdfFontPropData.iObjId <= 0) {
            pdfFontPropData.iObjId = iObjId++;
          }
          pdfPageData.RegisterFontProp(fontProp_String);
          pdfPageData.bProcSet_Text = true;
        }
        else {
          throw new ReportException("unknown report object type [" + repObj.GetType().FullName + "]"); 
        }
      }
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Creates the string of the specified matrix.</summary>
    /// <param name="page">Page of this point</param>
    /// <param name="m">Trasformation matrix</param>
    private String sMatrix(Page page, MatrixD m) {
      return RT.sPdfDim(m.rSX) + " " + RT.sPdfDim(-m.rRY) + " " + RT.sPdfDim(-m.rRX) + " " + RT.sPdfDim(m.rSY) + " " + sPoint(page, m.rDX, m.rDY);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Creates the string of the absolute coordinates of the specified point.</summary>
    /// <param name="page">Page of this point</param>
    /// <param name="rX">X</param>
    /// <param name="rY">Y</param>
    private String sPoint(Page page, Double rX, Double rY) {
      return RT.sPdfDim(rX) + " " + RT.sPdfDim(page.rHeight - rY);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Applies the geometric transform represented by the matrix to the point and creates the string of the absolute coordinates.</summary>
    /// <param name="page">Page of this point</param>
    /// <param name="m">Trasformation matrix</param>
    /// <param name="rX">X</param>
    /// <param name="rY">Y</param>
    private String sPoint(Page page, MatrixD m, Double rX, Double rY) {
      return sPoint(page, m.rTransformX(rX, rY), m.rTransformY(rX, rY));
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Writes a string to the buffer.</summary>
    /// <param name="s">String to append to the buffer</param>
    private void Write(String s) {
      sb.Append(s);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Writes the brush properties into the buffer.</summary>
    /// <param name="brushProp">New brush properties</param>
    private void Write_Brush(BrushProp brushProp) {
      Write_rg(brushProp.color);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Writes the font properties into the buffer.</summary>
    /// <param name="fontProp">New font properties</param>
    private void Write_Font(FontProp fontProp) {
      Write_rg(fontProp.color);
      PdfFontPropData pdfFontPropData = (PdfFontPropData)fontProp.fontPropData;
      String sFont_New = pdfFontPropData.iObjId + " " + RT.sPdfDim(pdfFontPropData.rSizeFactor(fontProp));
      if (sFont_Cur != sFont_New) {
      WriteLine("/" + sFontAbbr + sFont_New + " Tf");
      sFont_Cur = sFont_New;
      }
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Writes the pen properties into the buffer.</summary>
    /// <param name="penProp">New pen properties</param>
    private void Write_Pen(PenProp penProp) {
      if (!bColor_RG || !penProp.color.Equals(color_RG)) {
        if (penProp.color.R == penProp.color.G && penProp.color.G == penProp.color.B) {
          WriteLine(RT.sPdfDim(penProp.color.R / 255.0) + " G"); 
        }
        else {
          WriteLine(RT.sPdfDim(penProp.color.R / 255.0) + " " + RT.sPdfDim(penProp.color.G / 255.0)  + " " + RT.sPdfDim(penProp.color.B / 255.0) + " RG"); 
        }
        bColor_RG = true;
        color_RG = penProp.color;
      }
      if (rPenWidth_Cur != penProp.rWidth) {
        WriteLine(RT.sPdfDim(penProp.rWidth) + " w"); 
        rPenWidth_Cur = penProp.rWidth;
      }
      if (rPatternOn != penProp.rPatternOn || rPatternOff != penProp.rPatternOff) {
        if (penProp.rPatternOff == 0) {
          WriteLine("[] 0 d");
        }
        else {
          if (penProp.rPatternOn == penProp.rPatternOff) {
            WriteLine("[" + RT.sPdfDim(penProp.rPatternOn) + "] 0 d");
          }        
          else {
            WriteLine("[" + RT.sPdfDim(penProp.rPatternOn) + " " + RT.sPdfDim(penProp.rPatternOff) + "] 0 d");
          }        
        }
      }
    }

    //----------------------------------------------------------------------------------------------------
    /// <summary>Writes the background color into the buffer.</summary>
    /// <param name="color">New background color</param>
    private void Write_rg(Color color) {
      if (!bColor_rg || !color.Equals(color_rg)) {
        if (color.R == color.G && color.G == color.B) {
          WriteLine(RT.sPdfDim(color.R / 255F) + " g"); 
        }
        else {
          WriteLine(RT.sPdfDim(color.R / 255F) + " " + RT.sPdfDim(color.G / 255F) + " " + RT.sPdfDim(color.B / 255F) + " rg"); 
        }
        bColor_rg = true;
        color_rg = color;
      }
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Writes a string directly to the output stream.</summary>
    /// <param name="s">String to write to the output stream</param>
    private void WriteDirect(String s) {
      Byte[] aByte = System.Text.Encoding.Default.GetBytes(s);
      bufferedStream.Write(aByte, 0, aByte.Length);
      iBytesWrittenToStream += aByte.Length;
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Writes a line of text into the buffer.</summary>
    /// <param name="s">String to append to the buffer</param>
    private void WriteLine(String s) {
      sb.Append(s);
      sb.Append('\n');
    }

  }
}
