using System;

// Creation date: 05.11.2002
// Checked: 08.07.2003
// Author: Otto Mayer (mot@root.ch)
// Version: 1.02

// Report.NET copyright 2002-2004 root-software ag, Brglen Switzerland - O. Mayer, S. Spirig, R. Gartenmann, all rights reserved
// This library is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation, version 2.1 of the License.
// This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details. You
// should have received a copy of the GNU Lesser General Public License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA www.opensource.org/licenses/lgpl-license.html

namespace Root.Reports {
  /// <summary>Definition of the column properties</summary>
  /// <remarks>
  /// This class defines the properties of a column for the table layout manager.
  /// </remarks>
  public abstract class TlmColumnDef {
    //----------------------------------------------------------------------------------------------------x
    #region Border Margins
    //----------------------------------------------------------------------------------------------------x

    private Double _rBorderTop = Double.NaN;
    /// <summary>Gets or sets the top border margin of the column (default: <see cref="System.Double.NaN"/>).</summary>
    /// <value>
    /// The top border margin of the column in points (1/72 inch) or <see cref="System.Double.NaN"/>
    /// if the cell margin should not be overwritten.
    /// </value>
    /// <remarks>
    /// This value sets the top border margin of the column.
    /// It overwrites the top margin (<see cref="TlmCellDef.rMarginTop"/>) of the topmost cell.
    /// Tables often have different properties for the topmost and bottommost lines than for the other cells.
    /// If this value is <see cref="System.Double.NaN"/> the top margin of the topmost cell will not be overwritten.
    /// <para>For the metric version see <see cref="TlmColumnDef.rBorderTopMM"/>.</para>
    /// </remarks>
    public Double rBorderTop {
      get { return _rBorderTop; }
      set { _rBorderTop = value; }
    }

    /// <summary>Gets or sets the top border margin of the column (default: <see cref="System.Double.NaN"/>).</summary>
    /// <value>
    /// The top border margin of the column in millimeters or <see cref="System.Double.NaN"/>
    /// if the cell margin should not be overwritten.
    /// </value>
    /// <remarks>
    /// This value sets the top border margin of the column.
    /// It overwrites the top margin (<see cref="TlmCellDef.rMarginTopMM"/>) of the topmost cell.
    /// Tables often have different properties for the topmost and bottommost lines than for the other cells.
    /// If this value is <see cref="System.Double.NaN"/> the top margin of the topmost cell will not be overwritten.
    /// <para>For the inch version see <see cref="TlmColumnDef.rBorderTop"/>.</para>
    /// </remarks>
    public Double rBorderTopMM {
      get { return RT.rMMFromPoint(rBorderTop); }
      set { rBorderTop = RT.rPointFromMM(value); }
    }

    private Double _rBorderBottom = Double.NaN;
    /// <summary>Gets or sets the bottom border margin of the column (default: <see cref="System.Double.NaN"/>).</summary>
    /// <value>
    /// The bottom border margin of the column in points (1/72 inch) or <see cref="System.Double.NaN"/>
    /// if the cell margin should not be overwritten.
    /// </value>
    /// <remarks>
    /// This value sets the bottom border margin of the column.
    /// It overwrites the bottom margin (<see cref="TlmCellDef.rMarginBottom"/>) of the bottommost cell.
    /// Tables often have different properties for the topmost and bottommost lines than for the other cells.
    /// If this value is <see cref="System.Double.NaN"/> the bottom margin of the bottommost cell will not be overwritten.
    /// <para>For the metric version see <see cref="TlmColumnDef.rBorderBottomMM"/>.</para>
    /// </remarks>
    public Double rBorderBottom {
      get { return _rBorderBottom; }
      set { _rBorderBottom = value; }
    }

    /// <summary>Gets or sets the bottom border margin of the column (default: <see cref="System.Double.NaN"/>).</summary>
    /// <value>
    /// The bottom border margin of the column in millimeters or <see cref="System.Double.NaN"/>
    /// if the cell margin should not be overwritten.
    /// </value>
    /// <remarks>
    /// This value sets the bottom border margin of the column.
    /// It overwrites the bottom margin (<see cref="TlmCellDef.rMarginBottomMM"/>) of the bottommost cell.
    /// Tables often have different properties for the topmost and bottommost lines than for the other cells.
    /// If this value is <see cref="System.Double.NaN"/> the bottom margin of the bottommost cell will not be overwritten.
    /// <para>For the inch version see <see cref="TlmColumnDef.rBorderBottom"/>.</para>
    /// </remarks>
    public Double rBorderBottomMM {
      get { return RT.rMMFromPoint(rBorderBottom); }
      set { rBorderBottom = RT.rPointFromMM(value); }
    }

    /// <summary>Gets or sets the vertical border margins of the column.</summary>
    /// <value>
    /// The vertical border margins of the column in points (1/72 inch) or <see cref="System.Double.NaN"/>
    /// if the cell margins should not be overwritten.
    /// </value>
    /// <remarks>
    /// This value sets the vertical border margins of the column,
    /// i.e. the top <see cref="TlmColumnDef.rBorderTop"/> and bottom margins <see cref="TlmColumnDef.rBorderBottom"/>.
    /// <para>For the metric version see <see cref="TlmColumnDef.rBorderV_MM"/>.</para>
    /// </remarks>
    public Double rBorderV {
      get { return (rBorderTop + rBorderBottom) / 2; }
      set { rBorderTop = rBorderBottom = value; }
    }

    /// <summary>Gets or sets the vertical border margins of the column.</summary>
    /// <value>
    /// The vertical border margins of the column in millimeters or <see cref="System.Double.NaN"/>
    /// if the cell margins should not be overwritten.
    /// </value>
    /// <remarks>
    /// This value sets the vertical border margins of the column,
    /// i.e. the top <see cref="TlmColumnDef.rBorderTopMM"/> and bottom margins <see cref="TlmColumnDef.rBorderBottomMM"/>.
    /// <para>For the inch version see <see cref="TlmColumnDef.rBorderV"/>.</para>
    /// </remarks>
    public Double rBorderV_MM {
      get { return RT.rMMFromPoint(rBorderV); }
      set { rBorderV = RT.rPointFromMM(value); }
    }
    #endregion

    //----------------------------------------------------------------------------------------------------x
    #region Border Lines
    //----------------------------------------------------------------------------------------------------x

    private PenProp _pp_BorderTop = PenProp.pp_Null;
    /// <summary>Gets or sets the pen properties of the top border line of the column (default: <see langword="null"/>).</summary>
    /// <value>
    ///		<list type="table">
    ///			<listheader>
    ///				<term>Value</term>
    ///				<description>Description</description>
    ///			</listheader>
    ///			<item>
    ///				<term><see cref="Root.Reports.PenProp"/> object</term>
    ///				<description>The top border line of the topmost cell of the table will be overwritten by this value.</description>
    ///			</item>
    ///			<item>
    ///				<term><see langword="null"/></term>
    ///				<description>The top border line of the topmost cell of the table will be removed.</description>
    ///			</item>
    ///			<item>
    ///				<term><see cref="PenProp.pp_Null"/></term>
    ///				<description>This value prevents that the top border line of the topmost cell of the table will be overwritten.</description>
    ///			</item>
    ///		</list>
    /// </value>
    /// <remarks>
    /// This value sets the top line of the column.
    /// It overwrites the top line (<see cref="TlmCellDef.pp_LineTop"/>) of the topmost cell.
    /// Tables often have different properties for the topmost and bottommost lines than for the other cells.
    /// If this value is <see cref="PenProp.pp_Null"/> the top line of the cell will not be overwritten.
    /// </remarks>
    public PenProp pp_BorderTop {
      get { return _pp_BorderTop; }
      set { _pp_BorderTop = value; }
    }

    private PenProp _pp_BorderBottom = PenProp.pp_Null;
    /// <summary>Gets or sets the pen properties of the bottom border line of the column (default: <see langword="null"/>).</summary>
    /// <value>
    ///		<list type="table">
    ///			<listheader>
    ///				<term>Value</term>
    ///				<description>Description</description>
    ///			</listheader>
    ///			<item>
    ///				<term><see cref="Root.Reports.PenProp"/> object</term>
    ///				<description>The bottom border line of the bottommost cell of the table will be overwritten by this value.</description>
    ///			</item>
    ///			<item>
    ///				<term><see langword="null"/></term>
    ///				<description>The bottom border line of the bottommost cell of the table will be removed.</description>
    ///			</item>
    ///			<item>
    ///				<term><see cref="PenProp.pp_Null"/></term>
    ///				<description>This value prevents that the bottom border line of the bottommost cell of the table will be overwritten.</description>
    ///			</item>
    ///		</list>
    /// </value>
    /// <remarks>
    /// This value sets the bottom line of the column.
    /// It overwrites the bottom line (<see cref="TlmCellDef.pp_LineBottom"/>) of the bottommost cell.
    /// Tables often have different properties for the topmost and bottommost lines than for the other cells.
    /// If this value is <see cref="PenProp.pp_Null"/> the bottom line of the cell will not be overwritten.
    /// </remarks>
    public PenProp pp_BorderBottom {
      get { return _pp_BorderBottom; }
      set { _pp_BorderBottom = value; }
    }

    /// <summary>Sets the pen properties of the horizontal border lines of the column.</summary>
    /// <value>
    ///		<list type="table">
    ///			<listheader>
    ///				<term>Value</term>
    ///				<description>Description</description>
    ///			</listheader>
    ///			<item>
    ///				<term><see cref="Root.Reports.PenProp"/> object</term>
    ///				<description>The top border line of the topmost cell and the bottom border line of the bottommost cell of the table will be overwritten by this value.</description>
    ///			</item>
    ///			<item>
    ///				<term><see langword="null"/></term>
    ///				<description>The top border line of the topmost cell and the bottom border line of the bottommost cell of the table will be removed.</description>
    ///			</item>
    ///			<item>
    ///				<term><see cref="PenProp.pp_Null"/></term>
    ///				<description>This value prevents that the top border line of the topmost cell and the bottom border line of the bottommost cell of the table will be overwritten.</description>
    ///			</item>
    ///		</list>
    /// </value>
    /// <remarks>
    /// This value sets the horizontal lines of the column.
    /// It overwrites the top border line of the topmost cell and the bottom line (<see cref="TlmCellDef.pp_LineBottom"/>) of the bottommost cell.
    /// Tables often have different properties for the topmost and bottommost lines than for the other cells.
    /// If this value is <see cref="PenProp.pp_Null"/> the horizontal lines of the cells will not be overwritten.
    /// </remarks>
    public PenProp pp_BorderH {
      set { pp_BorderTop = pp_BorderBottom = value; }
    }
    #endregion

  }
}
