#region License
// Copyright (c) 2005 Griffin Caprio & Choy Rim. All rights reserved.
#endregion
#region Imports
using MbUnit.Core.Framework;
using MbUnit.Framework;

using DotNetMock;
using DotNetMock.Dynamic;
#endregion

namespace DotNetMock.Examples.Dynamic
{

	[TestFixture]
	public class WeatherTest
	{

		private IDynamicMock random;
		private Weather weather;
	
		[SetUp]
		public void SetUp() 
		{
			random = new DynamicMock(typeof(IWeatherRandom));
			weather = new Weather((IWeatherRandom)random.Object);			
		}
		
		[Test]	
		public void RandomRaining() 
		{
			random.SetValue("NextTemperature", 1.0);
			random.SetValue("NextIsRaining", true);
			weather.Randomize();
			Assertion.Assert("is raining", weather.IsRaining);
		}
	
		[Test]	
		public void RandomNotRaining() 
		{
			random.SetValue("NextTemperature", 1.0);
			random.SetValue("NextIsRaining", false);
			weather.Randomize();
			Assertion.Assert("is not raining", !weather.IsRaining);
		}
	
		[Test]
 		public void RandomTemperatureSunny() 
 		{
			double TEMPERATURE = 20.0;
			random.SetValue("NextTemperature", TEMPERATURE);
			random.SetValue("NextIsRaining", false);
			weather.Randomize();
			Assertion.AssertEquals("temperature", TEMPERATURE, weather.Temperature);
		}

		[Test]
 		public void RandomTemperatureRaining() 
 		{
			double TEMPERATURE = 20.0;
			random.SetValue("NextTemperature", TEMPERATURE);
			random.SetValue("NextIsRaining", true);
			weather.Randomize();
			Assertion.AssertEquals("temperature", TEMPERATURE / 2.0, weather.Temperature);
		}

	}

	[TestFixture]
	public class DefaultWeatherRandomTest
	{
		
		[Test]
		public void NextIsRaining()
		{
			IDynamicMock random = new DynamicMock(typeof(System.Random));
			IWeatherRandom weather = new DefaultWeatherRandom((System.Random)random.Object);
			
			random.SetValue("NextDouble", 0.0);
			Assertion.Assert("is raining", weather.NextIsRaining());
			
			random.SetValue("NextDouble", DefaultWeatherRandom.CHANCE_OF_RAIN);
			Assertion.Assert("is not raining", !weather.NextIsRaining());
			
			random.SetValue("NextDouble", 1.0);
			Assertion.Assert("is not raining", !weather.NextIsRaining());
		}
	
		[Test]
		public void NextTemperature()
		{
			IDynamicMock random = new DynamicMock(typeof(System.Random));
			IWeatherRandom weather = new DefaultWeatherRandom((System.Random)random.Object);
			
			random.SetValue("NextDouble", 0.0);
			Assertion.AssertEquals("should be min temperature", 
				DefaultWeatherRandom.MIN_TEMPERATURE,
				weather.NextTemperature()
			);
			
			random.SetValue("NextDouble", 0.5);
			Assertion.AssertEquals("should be average temperature", 
				0.5 * (DefaultWeatherRandom.MIN_TEMPERATURE + DefaultWeatherRandom.MAX_TEMPERATURE),
				weather.NextTemperature()
			);
			
			random.SetValue("NextDouble", 1.0);
			Assertion.AssertEquals("should be max temperature", 
				DefaultWeatherRandom.MAX_TEMPERATURE,
				weather.NextTemperature()
			);
		}
	
	}
}
