using System;
using System.Diagnostics;

// Creation date: 24.04.2002
// Checked: 18.07.2002
// Author: Otto Mayer (mot@root.ch)
// Version: 1.01

// Report.NET copyright 2002-2004 root-software ag, Brglen Switzerland - O. Mayer, S. Spirig, R. Gartenmann, all rights reserved
// This library is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation, version 2.1 of the License.
// This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details. You
// should have received a copy of the GNU Lesser General Public License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA www.opensource.org/licenses/lgpl-license.html

namespace Root.Reports {
  /// <summary>Font Definition</summary>
  /// <remarks>Each font family must be registered before it can be used. It can be registered only once.</remarks>
  /// <example>Font definition sample:
  /// <code>
  /// class HelloWorld {
  ///   public static void Main() {
  ///     Report report = new Report(new PdfFormatter());
  ///     <b>FontDef fd = new FontDef(report, FontDef.StandardFont.Helvetica);</b>
  ///     FontProp fp = new FontPropMM(fd, 25);
  ///     Page page = new Page(report);
  ///     page.AddCenteredMM(80, new RepString(fp, "Hello World!"));
  ///     RT.ViewPDF(report, "HelloWorld.pdf");
  ///   }
  /// }
  /// </code>
  /// </example>
  public class FontDef {
    //----------------------------------------------------------------------------------------------------x
    #region Static
    //----------------------------------------------------------------------------------------------------x

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Gets a font handle for the specified font family.</summary>
    /// <remarks>If the font family is already registered a ReportException will be thrown.</remarks>
    /// <param name="standardFont">Standard font enumeration value</param>
    private static String sGetFontName(StandardFont standardFont) {
      String sFontName = null;
      if (standardFont == StandardFont.TimesRoman) {
        sFontName = "Times-Roman";
      }
      else {
        sFontName = Enum.GetName(typeof(StandardFont), standardFont);
      }
      return sFontName;
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Gets a font handle for the specified font family.</summary>
    /// <remarks>If the font family is already registered a ReportException will be thrown.</remarks>
    /// <param name="report">Report to which this font belongs</param>
    /// <param name="sFontName">Name of the font family</param>
    public static FontDef fontDef_FromName(Report report, String sFontName) {
      Debug.Assert(sFontName != null);
      FontDef fontDef = (FontDef)report.ht_FontDef[sFontName];
      if (fontDef == null) {
        return new FontDef(report, sFontName);
      }
      return fontDef;
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Gets a font handle for the specified font family.</summary>
    /// <remarks>If the font family is already registered a ReportException will be thrown.</remarks>
    /// <param name="report">Report to which this font belongs</param>
    /// <param name="standardFont">Standard font enumeration value</param>
    public static FontDef fontDef_FromName(Report report, StandardFont standardFont) {
      return fontDef_FromName(report, sGetFontName(standardFont));
    }
    #endregion

    //====================================================================================================x
    /// <summary>The font is defined in this report.</summary>
    public readonly Report report;

    /// <summary>Name of the font family</summary>
    public readonly String sFontName;

    //internal FontData fontData_Standard;
    //internal FontData fontData_Bold;
    //internal FontData fontData_Italic;
    //internal FontData fontData_BoldItalic;

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Initializes a new font definition.</summary>
    /// <param name="report">Report to which this font belongs</param>
    /// <param name="sFontName">Name of the font family</param>
    /// <remarks>If the font family is already registered a ReportException will be thrown.</remarks>
    public FontDef(Report report, String sFontName) {
      this.report = report;
      this.sFontName = sFontName;

      if (report.ht_FontDef.Contains(sFontName)) {
        throw new ReportException("Font [" + sFontName + " ] is already defined");
      }
      report.ht_FontDef.Add(sFontName, this);

      report.formatter.InitFontDef(this);
    }

    //----------------------------------------------------------------------------------------------------x
    /// <summary>Initializes a new font definition.</summary>
    /// <param name="report">Report to which this font belongs</param>
    /// <param name="standardFont">Standard font enumeration value</param>
    /// <remarks>If the font family is already registered a ReportException will be thrown.</remarks>
    public FontDef(Report report, StandardFont standardFont) : this(report, sGetFontName(standardFont)) {
    }

    //====================================================================================================x
    /// <summary>Predefined standard fonts</summary>
    public enum StandardFont {
      /// <summary>Standard base 14 type 1 font "Courier"</summary>
      Courier,
      /// <summary>Standard base 14 type 1 font "Helvetica"</summary>
      Helvetica,
      /// <summary>Standard base 14 type 1 font "Symbol"</summary>
      Symbol,
      /// <summary>Standard base 14 type 1 font "Times-Roman"</summary>
      TimesRoman,
      /// <summary>Standard base 14 type 1 font "ZapfDingbats"</summary>
      ZapfDingbats
    }

  }
}
