using System;
using System.Collections;
using System.Security.Principal;
using DotNetMock;

namespace DotNetMock.Framework.Security.Principal
{
	/// <summary>
	/// Mock Object to implement the IPrincipal interface
	/// </summary>
	public class MockIPrincipal : MockObject, IPrincipal
	{
		/// <summary>
		/// Current Identity associated with this Principal
		/// </summary>
		private IIdentity _expectedIdentity = null;
		/// <summary>
		/// Expected number of IsInRole() calls
		/// </summary>
		private ExpectationCounter _isInRoleCalls = new ExpectationCounter("MockIPrincipal.IsIsRoleCalls");
		/// <summary>
		/// Lists of roles the current Principal belongs to
		/// </summary>
		private ArrayList _roles = null;
		/// <summary>
		/// Default Constructor
		/// </summary>
		public MockIPrincipal() : base("MockIPrincipal")
		{
			_roles = new ArrayList();
		}
		#region MockMethods
		/// <summary>
		/// Sets the expected value of Identity to return
		/// </summary>
		/// <param name="identity">Identity for current Principal</param>
		public void SetExpectedIdentity( IIdentity identity ) 
		{
			_expectedIdentity = identity;
		}
		/// <summary>
		/// Sets the number of calls to IsInRole()
		/// </summary>
		/// <param name="count">expected number of calls to IsInRol()</param>
		public void SetExpectedIsInRoleCount( int count )
		{
			_isInRoleCalls.Expected = count;	
		}
		/// <summary>
		/// Adds the given role to the list of roles this Principal belongs to
		/// </summary>
		/// <param name="role">role to add</param>
		public void AddExpectedRole( string role ) 
		{
			_roles.Add( role );
		}
		/// <summary>
		/// Adds the given roles to the list of roles this Principal belongs to
		/// </summary>
		/// <param name="roles"></param>
		public void AddExpectedRoles( string[] roles )
		{
			for (int i = 0; i < roles.Length; i++) 
			{
				AddExpectedRole( roles[i] );
			}
		}
		#endregion
		#region Implementation of IPrincipal
		/// <summary>
		/// Returns true/false if the current principal belongs to the given role
		/// </summary>
		/// <param name="roleToSearch">Role to check for</param>
		/// <returns>True/False</returns>
		public bool IsInRole(string roleToSearch)
		{
			_isInRoleCalls.Inc();
			bool found = false;
			foreach (string role in _roles) 
			{
				if (role.Equals(roleToSearch)) 
				{
					found = true;
				}
			}
			return found;
		}
		/// <summary>
		/// Returns the current Identity associated with this Principal
		/// </summary>
		public System.Security.Principal.IIdentity Identity
		{
			get
			{
				if (_expectedIdentity == null) 
				{
					return new MockIIdentity();
				}
				return _expectedIdentity;
			}
		}
		#endregion

	}
}
