namespace DotNetMock.Tests
{
	using System;
	using System.Collections;
	using NUnit.Framework;
	using DotNetMock;

	/// <summary>
	/// Summary description for ExpectationArrayListTests.
	/// </summary>

	[TestFixture]
	public class ExpectationArrayListTests
	{
		private ExpectationArrayList _expectationArrayList = null;
		
		[SetUp]
		public void SetUp()
		{
			_expectationArrayList = new ExpectationArrayList("ExpectationArrayList");
		}
		[TearDown]
		public void TearDown()
		{
			_expectationArrayList = null;
		}

		[Test]
		public void Empty()
		{
			_expectationArrayList.Verify();
		}
		[Test]
		public void HasExpectations()
		{
			ArrayList arrayList3 = new ArrayList();
			arrayList3.Add("A");

			Assert.IsTrue(!_expectationArrayList.HasExpectations, "Should not have expectations.");
			_expectationArrayList.AddExpectedMany(arrayList3);
			Assert.IsTrue(_expectationArrayList.HasExpectations, "Should have expectations.");
			
			arrayList3 = null;
		}
		
		[Test]
		public void HasExpectationsFromArray()
		{
			string[] strings = new String[3];
			strings[0] = "A";
			strings[1] = "B";
			strings[2] = "C";

			Assert.IsTrue(!_expectationArrayList.HasExpectations);
			_expectationArrayList.AddExpectedMany(strings);
			Assert.IsTrue(_expectationArrayList.HasExpectations);
		}

		[Test]
		public void HasExpectationsFromEnumerator()
		{
			Hashtable table = new Hashtable();
			table.Add("mock", "objects");

			Assert.IsTrue(!_expectationArrayList.HasExpectations);
			_expectationArrayList.AddExpectedMany(table);
			Assert.IsTrue(_expectationArrayList.HasExpectations);
		}

		[Test]
		public void HasNoExpectations()
		{
			_expectationArrayList.AddActual("A");
			Assert.IsTrue(!_expectationArrayList.HasExpectations);
		}
		[Test]
		public void Verify()
		{
			ArrayList arrayList4 = new ArrayList();
			ArrayList arrayList5 = new ArrayList();
			ArrayList arrayList6 = new ArrayList();
			
			arrayList4.Add(3);
			arrayList4.Add(4);
			arrayList4.Add(5);

			arrayList5.Add(6);
			arrayList5.Add(7);
			arrayList5.Add(8);

			arrayList6.Add(3);
			arrayList6.Add(4);
			arrayList6.Add(5);

			_expectationArrayList.AddActualMany(arrayList4);
			_expectationArrayList.AddExpectedMany(arrayList5);
			try 
			{
				_expectationArrayList.Verify();
				Assert.Fail("Should have thrown an exception.");
			}
			catch
			{
			}
			_expectationArrayList.ClearExpected();
			_expectationArrayList.AddExpectedMany(arrayList6);
			_expectationArrayList.Verify();

			arrayList4 = null;
			arrayList5 = null;
			arrayList6 = null;
		}
		
		[Test]
		[ExpectedException(typeof(DotNetMock.AssertionException))]
		public void FailImmediately()
		{
			_expectationArrayList.AddExpected("A");
			_expectationArrayList.AddExpected("B");
			_expectationArrayList.VerifyImmediate = true;

			_expectationArrayList.AddActual("A");
			_expectationArrayList.AddActual("C");
		}
		[Test]
		[ExpectedException(typeof(DotNetMock.AssertionException))]
		public void FailImmediatelyTooMany()
		{
			_expectationArrayList.AddExpected("A");
			_expectationArrayList.AddActual("A");
			_expectationArrayList.VerifyImmediate = true;
			_expectationArrayList.AddActual("C");
		}

		[Test]
		public void FlushExpected()
		{
			_expectationArrayList.AddExpected("A");
			_expectationArrayList.ExpectNothing();
			Assert.IsTrue( _expectationArrayList.HasExpectations );
			_expectationArrayList.AddActual( "A" );
			try 
			{
				_expectationArrayList.Verify();
				Assert.Fail( "Should have thrown an exception" );
			} 
			catch {}
		}

		[Test]
		[ExpectedException(typeof(DotNetMock.AssertionException))]
		public void FailingVerify()
		{
			_expectationArrayList.AddExpected("A");
			_expectationArrayList.AddExpected("B");
			_expectationArrayList.AddExpected("C");

			_expectationArrayList.AddActual("A");
			_expectationArrayList.AddActual("B");
			_expectationArrayList.AddActual("B");
			
			_expectationArrayList.Verify();
		}

		[Test]
		public void AddSingleItem()
		{
			int int1 = 2;
			
			_expectationArrayList.AddExpected(int1);
			_expectationArrayList.AddActual(int1);
			
			_expectationArrayList.Verify();
		}
		[Test]
		public void AddManyFromArray()
		{
			string[] array = new string[3];
			array[0] = "A";
			array[1] = "B";
			array[2] = "C";

			_expectationArrayList.AddExpectedMany(array);
			_expectationArrayList.AddActualMany(array);
			_expectationArrayList.Verify();
		}
		[Test]
		public void AddManyFromEnumerator()
		{
			Hashtable hashExpected = new Hashtable();
			hashExpected.Add("mock", "objects");
			hashExpected.Add("source", "forge");

			Hashtable hashActual = new Hashtable();
			hashActual.Add("mock", "objects");
			hashActual.Add("source", "forge");

			_expectationArrayList.AddExpectedMany(hashExpected);
			_expectationArrayList.AddActualMany(hashActual);
			_expectationArrayList.Verify();
		}
	}
}
