using Root.Reports;
using System;
using System.IO;
using System.Data.OleDb;
using System.Drawing;

// Creation date: 08.11.2002
// Checked: 31.10.2004
// Author: Otto Mayer (mot@root.ch)
// Version: 1.03

// Report.NET copyright 2002-2004 root-software ag, Brglen Switzerland - Otto Mayer, Stefan Spirig, all rights reserved
// This library is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License

// as published by the Free Software Foundation, version 2.1 of the License.

// This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details. You

// should have received a copy of the GNU Lesser General Public License along with this library; if not, write to the Free Software

// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA www.opensource.org/licenses/lgpl-license.html

namespace ReportSamples {
  /// <summary>Table Layout Manager and ADO.NET Sample</summary>
  /// <remarks>
  /// This sample creates a table with data from an ADO.NET data source (SQL-Server, sample database "Northwind").
  /// The table layout manager <see cref="Root.Reports.TableLayoutManager"/> automatically builds the grid lines,
  /// header of the table, page breaks, etc.
  /// The event handler <see cref="ReportSamples.TableLayoutManagerSample.Tlm_NewContainer"/> binds each table container to a new page.
  /// The first page has a caption. The following pages have no caption and therefore the table can be made higher.
  /// <note type="caution">
  ///   The connection string must contain the correct login data in order that the program can open a connection to the database.
  ///   <p/>Sample with <i>user id</i> "sa" and <i>password</i> "empty":
  ///   <code>private String sConnectionString = @"Provider=SQLOLEDB;initial catalog=northwind;user id=sa;password=";</code>
  /// </note>
  /// <para>PDF file: <see href="http://web.root.ch/developer/report_net/samples/TableLayoutManagerSample.pdf">TableLayoutManagerSample.pdf</see></para>
  /// <para>Source: <see href="http://web.root.ch/developer/report_net/samples/TableLayoutManagerSample.cs">TableLayoutManagerSample.cs</see></para>
  /// </remarks>
  public class TableLayoutManagerSample : Report {
    //private String sConnectionString = @"Provider=SQLOLEDB.1;Integrated Security=SSPI;Initial Catalog=Northwind;Data Source=(local)";
    private String sConnectionString = @"Provider=SQLOLEDB;initial catalog=northwind;user id=sa;password=";
    private FontDef fd;
    private Double rPosLeft = 20;
    private Double rPosRight = 195;
    private Double rPosTop = 24;
    private Double rPosBottom = 278;

    //------------------------------------------------------------------------------------------30.10.2004
    /// <summary>Creates this report.</summary>
    /// <remarks>
    /// This method overrides the <see cref="Root.Reports.Report.Create"/> method of the base class <see cref="Root.Reports.Report"/>.
    /// </remarks>
    protected override void Create() {  
      fd = new FontDef(this, "Arial");
      FontProp fp = new FontPropMM(fd, 1.9);  // standard font
      FontProp fp_Header = new FontPropMM(fd, 1.9);  // font of the table header
      fp_Header.bBold = true;

      Stream stream_Phone = GetType().Assembly.GetManifestResourceStream("ReportSamples.Phone.jpg");
      Random random = new Random(6);

      // create table
      using (TableLayoutManager tlm = new TableLayoutManager(fp_Header)) { 
        tlm.rContainerHeightMM = rPosBottom - rPosTop;  // set height of table
        tlm.headerCellDef.rAlignV = RepObj.rAlignCenter;  // set vertical alignment of all header cells
        tlm.cellDef.pp_LineBottom = new PenProp(this, 0.05, Color.LightGray);  // set bottom line for all cells
        tlm.eNewContainer += new TableLayoutManager.NewContainerEventHandler(Tlm_NewContainer);

        // define columns
        TableLayoutManager.Column col;
        col = new TableLayoutManager.ColumnMM(tlm, "ID", 13);

        col = new TableLayoutManager.ColumnMM(tlm, "Company Name", 40);
        col.cellDef.textMode = TableLayoutManager.TextMode.MultiLine;

        col = new TableLayoutManager.ColumnMM(tlm, "Address", 36);

        col = new TableLayoutManager.ColumnMM(tlm, "City", 22);

        col = new TableLayoutManager.ColumnMM(tlm, "Postal Code", 16);

        col = new TableLayoutManager.ColumnMM(tlm, "Country", 18);

        TableLayoutManager.Column col_Phone = new TableLayoutManager.ColumnMM(tlm, "Phone", rPosRight - rPosLeft - tlm.rWidthMM);
        col_Phone.fp_Header = new FontPropMM(fd, 1.9, Color.Brown);
        col_Phone.headerCellDef.rAlignH = RepObj.rAlignRight;
        col_Phone.cellDef.rAlignH = RepObj.rAlignRight;
        BrushProp bp_Phone = new BrushProp(this, Color.FromArgb(255, 255, 200));
        col_Phone.cellDef.bp_Back = bp_Phone;

        // open database
        using (OleDbConnection oleDbConnection = new OleDbConnection(sConnectionString)) {
          oleDbConnection.Open();
          OleDbCommand oleDbCommand = new OleDbCommand("SELECT CustomerID, CompanyName, Address, City, PostalCode, Country, Phone " +
            "FROM Customers ORDER BY CompanyName", oleDbConnection);
          // read rows
          using (OleDbDataReader oddr = oleDbCommand.ExecuteReader()) {
            BrushProp bp_USA = new BrushProp(this, Color.FromArgb(255, 180, 180));
            while (oddr.Read()) {
              tlm.cellDef.bp_Back = (oddr.GetString(5) == "USA" ? bp_USA : null);
              col_Phone.cellDef.bp_Back = (oddr.GetString(5) == "USA" ? new BrushProp(this, Color.FromArgb(255, 227, 50)) : bp_Phone);
              tlm.NewRow();
              tlm.Add(0, new RepString(fp, oddr.GetString(0)));
              tlm.Add(1, new RepString(fp, oddr.GetString(1)));
              tlm.Add(2, new RepString(fp, oddr.GetString(2)));
              tlm.Add(3, new RepString(fp, oddr.GetString(3)));
              if (!oddr.IsDBNull(4)) {
                tlm.Add(4, new RepString(fp, oddr.GetString(4)));
              }
              tlm.Add(5, new RepString(fp, oddr.GetString(5)));
              tlm.Add(6, new RepString(fp, oddr.GetString(6)));
              if (random.NextDouble() < 0.2) {  // mark randomly selected row with a phone icon
                tlm.tlmRow_Cur.aTlmCell[col_Phone].AddMM(1, 0.25, new RepImageMM(stream_Phone, 2.1, 2.3));
              }
            }
          }
        }
      }

      // print page number and current date/time
      foreach (Page page in enum_Page) {
        Double rY = rPosBottom + 1.5;
        page.AddLT_MM(rPosLeft, rY, new RepString(fp, DateTime.Now.ToShortDateString()  + "  " + DateTime.Now.ToShortTimeString()));
        page.AddRT_MM(rPosRight, rY, new RepString(fp, page.iPageNo + " / " + iPageCount));
      }
    }

    //------------------------------------------------------------------------------------------30.10.2004
    /// <summary>Creates a new page.</summary>
    /// <param name="oSender">Sender</param>
    /// <param name="ea">Event argument</param>
    /// <remarks>
    /// The first page has a caption. The following pages have no caption and therefore the table can be made higher.
    /// </remarks>
    public void Tlm_NewContainer(Object oSender, TableLayoutManager.NewContainerEventArgs ea) {  // only "public" for NDoc, should be "private"
      new Page(this);

      // first page with caption
      if (page_Cur.iPageNo == 1) {
        FontProp fp_Title = new FontPropMM(fd, 7);
        fp_Title.bBold = true;
        page_Cur.AddCT_MM(rPosLeft + (rPosRight - rPosLeft) / 2, rPosTop, new RepString(fp_Title, "Customer List"));
        ea.container.rHeightMM -= fp_Title.rLineFeedMM;  // reduce height of table container for the first page
      }

      // the new container must be added to the current page
      page_Cur.AddMM(rPosLeft, rPosBottom - ea.container.rHeightMM, ea.container);
    }
  }
}
